! makehtmshire.f90
! performs highly specialised operation to make set of internet files
! f90 -o makehtmshire filenames.f90 glofiles.f90 refregfiles.f90 makehtmshire.f90

program MakeHTMShire

use FileNames
use GloFiles
use RefRegFiles

implicit none

!*******************************************************************************
! definitions

real, pointer, dimension (:,:,:)		:: Data				! region,variable,period
real, pointer, dimension (:)			:: GloVec			! region

integer, pointer, dimension (:,:) 		:: MapIDLReg  			! shape of IDL mapping arrays
integer, pointer, dimension (:) 		:: RegSizes 			! region sizes + reg-->raw

character (len=20), pointer, dimension (:) 	:: RegNames			! names of individual regions
character (len=80), pointer, dimension (:) 	:: InFile, OutFile

character (len=3), dimension (17) :: PerSub					! names of time periods
character (len=3), dimension ( 5) :: VarSub					! names of variables
character (len=7), dimension ( 5) :: VarText					! text for variables

real, parameter :: MissVal = -999.0

integer :: ReadStatus, AllocStat
integer :: FileN,PerN,VarN,RegN
integer :: XFile, XVar, XPer, XReg, XChar
integer :: QOperation, QFinish, QVar, QPer
integer :: PerSubBeg, VarSubBeg, NumSubBeg, NumSubLen, TemplateLen, StrLen

character (len=80), parameter :: Blank = ""

character (len=80) :: GivenFile, NameOnly, Template, String, HTMCountry, HTMTable
character (len=20) :: NumSub

!*******************************************************************************
! main program

print*
print*, "  > ***** MakeHTMShire: performs specialised op to make internet files *****"
print*

print*, "  > Select the .glo files."
call GetBatch (Blank,InFile)
FileN = size (InFile, 1)
PerN  = 17
VarN  = nint (real(FileN) / real(PerN))

print*, "  > Identify the region set (/cru/u2/f709762/goglo/ref/)."
call LoadRefReg (Blank,MapIDLReg,RegSizes,RegNames)
RegN = size (RegSizes,1)

PerSub =['ann','MAM','JJA','SON','DJF','Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
VarSub =['tmp','dtr','pre','vap','cld']
VarText=['T mean ','T range','precip ','vapour ','cloud  ']

allocate (Data    (RegN,VarN,PerN), &
	  OutFile (RegN),           stat=AllocStat)
if (AllocStat.NE.0) print*, "  > ##### ERROR: MakeHTMShire: Data: Allocation failure #####"
Data = MissVal ; OutFile = 'missing'

do XFile = 1, FileN
  print*
  print "(a,i4)", "   > Loading file ", XFile
  
  GivenFile = InFile(XFile)

  call LoadGloVec (GivenFile, MapIDLReg, GloVec)
  call FindTimePer
  call FindVariable
  
  do XReg = 1, RegN
    Data (XReg,QVar,QPer) = GloVec (XReg)
  end do

  deallocate (GloVec,stat=AllocStat)
  if (AllocStat.NE.0) print*, "  > ##### ERROR: MakeHTMShire: Deallocation failure #####"
end do

deallocate (InFile,stat=AllocStat)
if (AllocStat.NE.0) print*, "  > ##### ERROR: MakeHTMShire: Deallocation failure #####"

call GetHTMPaths
call SaveHTM

deallocate (OutFile,Data,MapIDLReg,RegSizes,RegNames,stat=AllocStat)
if (AllocStat.NE.0) print*, "  > ##### ERROR: MakeHTMShire: Deallocation failure #####"

print*

contains

!*******************************************************************************
! get period index

subroutine FindTimePer

XPer = 0 ; QPer = 0
do
  XPer = XPer + 1
  
  PerSubBeg = index(GivenFile,PerSub(XPer))
  if (PerSubBeg.GT.0) QPer = XPer 
  
  if (XPer.EQ.PerN.OR.QPer.GT.0) exit
end do

end subroutine FindTimePer

!*******************************************************************************
! get variable index

subroutine FindVariable

XVar = 0 ; QVar = 0
do
  XVar = XVar + 1
  
  VarSubBeg = index(GivenFile,VarSub(XVar))
  if (VarSubBeg.GT.0) QVar = XVar 
  
  if (XVar.EQ.VarN.OR.QVar.GT.0) exit
end do

end subroutine FindVariable

!*******************************************************************************
! get .htm filepaths

subroutine GetHTMPaths

print*
print*, "  > Enter the filepath for saving the first region (include '0001'): "
Template    = SavePath (Blank,'.htm')
TemplateLen = len_trim(adjustl(Template))
OutFile (1) = Template
NumSubBeg   = index(Template,'0001')

do XReg = 2, RegN
  NumSub    = GetTextFromInt (XReg)  
  NumSubLen = len_trim(adjustl(NumSub))
  GivenFile = Template (1:(NumSubBeg-1))
  if (NumSubLen.LT.4) then
    do XChar = 1, (4-NumSubLen)
      GivenFile = trim(adjustl(GivenFile)) // '0'
    end do
  end if
  GivenFile = trim(adjustl(GivenFile)) // trim(adjustl(NumSub)) // Template ((NumSubBeg+4):TemplateLen)
  OutFile (XReg) = GivenFile
end do

end subroutine GetHTMPaths

!*******************************************************************************
! save each .htm in turn

subroutine SaveHTM

print*
print*, "  > Saving regions to .htm files..."

HTMCountry = '/cru/u2/f709762/goglo/f90/ref/htm-country.txt'
HTMTable   = '/cru/u2/f709762/goglo/f90/ref/htm-table.txt'

do XReg = 1, RegN
  open  (4, file=HTMCountry, status="replace", action="write", iostat=ReadStatus)
  write (4,"(a)"), trim(adjustl(RegNames(XReg)))
  close (4)

  open  (4, file=HTMTable, status="replace", action="write", iostat=ReadStatus)

  do XVar = 1, VarN
    String = '<p><tt><font size=-1>' // trim(adjustl(VarText(XVar)))
    StrLen = len_trim(VarText(XVar))      
    if (StrLen.LT.8) then
        do XChar = 1, (8-StrLen)
          String = trim(String) // '&nbsp;'
        end do
    end if
    write (4,"(a)"), String        
    
    do XPer = 1, PerN
      String = GetTextFromReal (Data(XReg,XVar,XPer))
      StrLen = len_trim(String)      
      if (StrLen.LT.5) then
        do XChar = 1, (5-StrLen)
          String = '&nbsp;' // trim(String)
        end do
      end if
      if (XPer.EQ.2.OR.XPer.EQ.6) String = '&nbsp;&nbsp;&nbsp;' // trim(String)
      write (4,"(a)"), String
    end do
    
    write (4,"(a)"), '</font></tt>'    
  end do
  
  close (4)

  call system ( 'cat /cru/u2/f709762/goglo/f90/ref/htmshire-part1.txt ' // HTMCountry // &
  		   ' /cru/u2/f709762/goglo/f90/ref/htmshire-part2.txt ' // HTMCountry // &
  		   ' /cru/u2/f709762/goglo/f90/ref/htmshire-part3.txt ' // HTMCountry // &
  		   ' /cru/u2/f709762/goglo/f90/ref/htmshire-part4.txt ' // HTMCountry // &
  		   ' /cru/u2/f709762/goglo/f90/ref/htmshire-part5.txt ' // HTMCountry // &
  		   ' /cru/u2/f709762/goglo/f90/ref/htmshire-part6.txt ' // HTMTable   // &
  		   ' /cru/u2/f709762/goglo/f90/ref/htmshire-part7.txt > ' // OutFile(XReg) )
end do

end subroutine SaveHTM

!*******************************************************************************
! end

end program MakeHTMShire
