#!/usr/bin/perl

# A program to calculate, for the 159 GLOSS-LTT tide stations in NOAA's list,
# a geographically-weighted average linear Mean Sea Level trend, using
# weightings determined by geographic proximity to other GLOSS-LTT tide
# stations.
#
# Here's NOAA's GLOSS-LTT list:
# http://tidesandcurrents.noaa.gov/sltrends/MSL_global_trendtable.html
#
# Run this program with no parameters for help.
#
# This program is compatible both with 32-bit Perl 4.036, and Perl 5.
#
# Copyright 2010, by David A. Burton
# Cary, NC  USA
# +1 919-481-0149
# Email: http://www.burtonsys.com/email/


# immediate output of debug prints
$| = 1;

# @INC = ('c:\html\climategate','.');

# print "before 'require splitcsv.pl'\n";
require "splitcsv.pl";

@sv_ARGV = @ARGV;



# These field numbers can be overridden by command-line options.
$fn_name = 1;  # first field is normally the name of the station
$fn_trend = 5;  # 5th field is normally the MSL trend for the station
$fn_lat = 13;  # 13th field is normally the latitude of the station
$fn_lon = 14;  # 14th field is normally the longitude of the station

$debugmode=0;  # for debug prints
while (($#ARGV >= 0) && ('-' eq substr($ARGV[0],0,1))) {
   if ('-d' eq $ARGV[0]) {
      $debugmode++;  # turn on debug prints
   } elsif ('-name=' eq substr($ARGV[0],0,6)) {
      $fn_name = substr($ARGV[0],6) + 0;
   } elsif ('-trend=' eq substr($ARGV[0],0,7)) {
      $fn_trend = substr($ARGV[0],7) + 0;
   } elsif ('-lat=' eq substr($ARGV[0],0,5)) {
      $fn_lat = substr($ARGV[0],5) + 0;
   } elsif ('-lon=' eq substr($ARGV[0],0,5)) {
      $fn_lon = substr($ARGV[0],5) + 0;
   } else {
      printf "ERROR: unrecognized command-line option: '%s'\n", $ARGV[0];
      exit 1;
   }
   shift @ARGV;
}

if ($debugmode) {
   print "dbg:  cmd='perl $0 " . join(' ', @sv_ARGV) . "'\n";
}


# some needed constants:
$pi = 3.1415926535897932384626433832795;
$pi_over_180 = $pi / 180.0;  # 0.017453292519943295769236907684886;
$radius_of_earth = 6366707.0;  # in meters


$num_args = $#ARGV+1;

if ($num_args != 1) {
 print "calculate_distance_weighted_msl_avg.pl --\n" .
       "   Calculate distance-weighted global linear Mean Sea Level (MSL) average\n" .
       "   from GLOSS-LTT tide stations data.\n" .
       "\n" .
       "Typical usage:\n" .
       "   perl $0 MSL_global_trendtable.csv\n" .
       "\n" .
       "Or, with options specified:\n" .
       "   perl $0 {options} MSL_global_trendtable.csv\n" .
       "\n" .
       "E.g., for debug prints:\n" .
       "   perl $0 -d MSL_global_trendtable.csv\n" .
       "\n" .
       "'MSL_global_trendtable.csv' is processed as follows:\n" .
       "\n" .
       "1) Read all 159 records, noting the Mean Sea Level (MSL) trend for\n" .
       "   each station and the station's coordinates.\n" .
       "2) For each station location, check it against every other location, calculate the\n" .
       "   distance D between the two locations, and the correlation weight W calculated\n" .
       "   from the distance.  Then (unless W=0) add the name/distance/weight triple\n" .
       "   to the list of nearby stations for that station location.\n" .
       "3) Also, for each location, sum the correlation weights of the nearby locations.\n" .
       "4) For each station, calculate its own weight W_own (to a maximum of 1.0) from the\n" .
       "   summed weights of the nearby stations, according to the formula:\n" .
       "     W_own = 1.0 / (1.0 + sum(nearby_weights))\n" .
       "   Thus, an isolated station has a weight of 1.0, but a station near other stations\n" .
       "   has a lower weight; e.g., a station which is very near just 1 other station has a\n" .
       "   weight of about 0.5.\n" .
       "5) Finally, calculate the geographically weighted average linear MSL trend from all\n" .
       "   159 stations, and display it.\n" .
       "\n" .
       "The following options are supported:\n" .
       "  -name=xx  (where xx is field number of the station name, default xx=1)\n" .
       "  -trend=xx (where xx is field number of the MSL trend, default xx=5)\n" .
       "  -lat=xx   (where xx is field number of the latitude, default xx=13)\n" .
       "  -lon=xx   (where xx is field number of the longitude, default xx=14)\n" .
       "  -d        (enable [massive!] debug prints)\n" .
       "\n" .
       "Dave Burton\n" .
       "Cary, NC  USA\n" .
       "+1-919-481-0149\n" .
       "\n";
       exit 1;
}

$inpfile = $ARGV[0];
if (!open(INP,"<$inpfile")) {
   printf "ERROR: could not open '%s', $!\n", $inpfile;
   exit 1;
}
print "reading '$inpfile' . . .\n";


# Test a string to see if it looks like a Degrees/Minutes latitude
# or longitude.  Result is 'LAT' or 'LON' or undef.
sub is_DM {
  local( $fld ) = shift;
  local( $result );
  undef $result;
  local( $deg, $min, $dir );
  if ((defined $fld) && ($fld =~ /^([0-9]+)[ \,]([0-9][0-9\.]*)( |)([NSEW])$/)) {
     # found a latitude or longitude
     $deg = $1;
     $min = $2;
     # I should test whether $deg,$min is in [-90..90] for latitude
     # or in [-180..180] for longitude, but I didn't bother.
     $dir = $4;
     if ($dir =~ /[NS]/) {
        $result = "LAT";
     } else {
        $result = "LON";
     }
  }
  return $result;
}


# Convert a Degrees,Minutes representation of latitude or longitude
# to decimal degrees.
sub DegMin_to_DecimalDeg {
  local( $fld ) = shift;
  local( $deg, $min, $dir );
  if ($fld =~ /^([0-9]+)[ \,]([0-9][0-9\.]*)( |)([NSEW])$/) {
     # found a latitude or longitude
     $deg = $1;
     $min = $2;
     $dir = $4;
     $deg = $1 + ($2 / 60);
     if ($dir =~ /[SW]/) {
        $deg = -$deg;
     }
     # if (length($deg) > 10) {
     #    $deg = sprintf("%4.5f",$deg);
     # }
  } else {
     print STDERR "ERR: DegMin_to_DecimalDeg(\"$fld\")  (not a recognized latitude or longitude)\n";
     undef $deg;
  }
  return $deg;
}



$num_recs = 0;
# Read the spreadsheet records (.csv format) into four arrays:
#   @rec_name (names of the stations)
#   @rec_trend (MSL trend at each station)
#   @rec_lat (latitude of each station)
#   @rec_lon (longitude of each station)
# $num_recs = number of stations (should be 159)
while (<INP>) {
   chop;
   $_ =~ s/[\r\n]*$//;  # because Perl 4 lacks chomp (to strip both cr and lf), make sure here

   # Split into fields.
   @sfields = &split_csv( $_ );
   $num_fields_with_content = 0;
   $first_field_is_only_field = 1;
   $fieldnumber = 0;
   foreach $fld (@sfields) {
      # get rid of leading and trailing whitespace in each field:
      if ($fld =~ /\s/) {
         $fld =~ s/^\s*//;  # strip leading whitespace, too.
         $fld =~ s/\s*$//;  # strip trailing blanks, tabs, cr, lf, etc.
         # get rid of any tabs, and collapse multiple spaces to one space:
         if ($fld =~ /\t|\s\s/) {
            $fld =~ s/\s+/ /g;
         }
      }
      if ('' ne $fld) {
         $num_fields_with_content++;
         if ($fieldnumber > 0) {
            $first_field_is_only_field = 0;
         }
      }

      # # upper-case each field value
      # $fld =~ tr/a-z/A-Z/;

      $fieldnumber++;
   }
   if (!$num_fields_with_content) {
      next;  # skip empty records
   }

   # 1st field is normally the name of the station
   # 5th field is normally the MSL trend for the station
   # 13th field is normally the latitude of the station
   # 14th field is normally the longitude of the station
   # These field numbers can be overridden by command-line options.

   if ( (&is_DM($sfields[$fn_lat-1]) ne 'LAT') ||
        (&is_DM($sfields[$fn_lon-1]) ne 'LON') ) {
      printf STDERR "ERR: record no. %d ('%s') lacks Lat/Lon in fields $fn_lat-$fn_lon: F$fn_lat='', F$fn_lon=''.\n", $num_recs, $sfields[0], $sfields[$fn_lat-1], $sfields[$fn_lon-1];
      exit 1;
   }
   $rec_name[$num_recs] = $sfields[$fn_name-1];
   $rec_trend[$num_recs] = $sfields[$fn_trend-1] + 0.0;
   $rec_lat[$num_recs] = &DegMin_to_DecimalDeg($sfields[$fn_lat-1]);
   $rec_lon[$num_recs] = &DegMin_to_DecimalDeg($sfields[$fn_lon-1]);
   $num_recs++;
}
print "$num_recs station records were read from $inpfile\n";
close INP;


# Compute distance between two locations via the Haversine Formula.
# (per http://mathforum.org/library/drmath/view/51879.html)
# Input is the two (latitude,longitude) pairs, in degrees.
# Output is the distance between the two locations, in meters.
# Formula is:
#   dlon = lon2 - lon1
#   dlat = lat2 - lat1
#   a = (sin(dlat/2))^2 + cos(lat1) * cos(lat2) * (sin(dlon/2))^2
#   c = 2 * atan2(sqrt(a), sqrt(1-a))
#   d = R * c
sub distance {
   local( $lat1, $lon1, $lat2, $lon2 ) = @_;
   # print "dbg: calculate distance between ($lat1,$lon1) and ($lat2,$lon2)\n";
   local( $result ) = 0.0;
   # first, convert degrees to radians:
   $lat1 *= $pi_over_180;
   $lon1 *= $pi_over_180;
   $lat2 *= $pi_over_180;
   $lon2 *= $pi_over_180;
   # then calcuate according to the Haversine Formula:
   local( $dlon ) = $lon2 - $lon1;
   local( $dlat ) = $lat2 - $lat1;
   local( $a1 ) = sin( $dlat/2.0 );
   $a1 = $a1 * $a1;
   local( $a2 ) = sin( $dlon/2.0 );
   $a2 = $a2 * $a2;
   local( $a ) = $a1 + (cos($lat1) * cos($lat2) * $a2);
   local( $c ) = 2.0 * atan2( sqrt($a), sqrt(1.0-$a) );
   $result = $radius_of_earth * $c;
   return $result;
}


## These are some sanity checks, which I used to verify that &distance
## works correctly (it does):
#
# $d1 = &distance( 0.0, 0.0, 0.0, 1.0 );
# # should be 1/360 of the circumference of the earth, which is
# # (1/360) * 40,003,200 meters = 111,120 meters.
# print " $d1 should be about 111,120 meters.\n";
#
# $d1 = &distance( 0.0, -1.0, 0.0, 0.0 );
# # should be 1/360 of the circumference of the earth, which is
# # (1/360) * 40,003,200 meters = 111,120 meters.
# print " $d1 should be about 111,120 meters.\n";
#
# $d1 = &distance( 0.0, -1.0, 0.0, 1.0 );
# # should be 2/360 of the circumference of the earth, which is
# # (2/360) * 40,003,200 meters = 222,240 meters.
# print " $d1 should be about 222,240 meters.\n";
#
# $d1 = &distance( 0.0, 1.0, 0.0, -1.0 );
# # should be 2/360 of the circumference of the earth, which is
# # (2/360) * 40,003,200 meters = 222,240 meters.
# print " $d1 should be about 222,240 meters.\n";
#
# $d1 = &distance( 0.0, -90.0, 0.0, 90.0 );
# # should be 1/2 of the circumference of the earth, which is
# # (1/2) * 40,003,200 meters = 20,001,600 meters.
# print " $d1 should be about 20,001,600 meters.\n";
#
# $d1 = &distance( 90.0, 0.0, -90.0, 0.0 );
# # should be 1/2 of the circumference of the earth, which is
# # (1/2) * 40,003,200 meters = 20,001,600 meters.
# print " $d1 should be about 20,001,600 meters.\n";
#
# $d1 = &distance( 0.0, 0.0, 0.0, 0.000009 );
# # should be about 1 meter.
# print " $d1 should be about 1 meter.\n";
#
# $d1 = &distance( 0.000009, 0.0, 0.0, 0.000009 );
# # should be about sqrt(2) = 1.414 meter.
# print " $d1 should be about 1.414 meter.\n";



# Distances for 0.33 weight and 0.00 weight, respectively:
$km400 = 400.00;
$km800 = 800.00;

# Linear approximating function for the measured/graphed correlation between
# distance and MSL Trend:
#
#   D     Y   Y_norm    W
#  ---  ----  ------  ----
#    0  0.00   0.00   1.00
#  400  2.00   0.67   0.33
#  800  3.00   1.00   0.00
#
# Input is distance in km; output is a number between 0 and 1.
sub dist_to_weight {
  local( $D ) = @_;
  local( $Y_norm, $W );
   if ($D <= $km400) {
      $Y = 2 * ($D/$km400);
   }
   if (($D > $km400) & ($D < $km800)) {
      $Y = 2 + (($D-$km400)/($km800-$km400));
   }
   if ($D >= $km800) {
      $Y = 3;
   }
   $Y_norm = $Y / 3;
   $W = 1 - $Y_norm;
   return $W;
}


# trim off country names, for conciseness
sub city_name_only {
   local($name) = shift;
   $name =~ s/[\,\/].*$//;
   if ("Galveston Pier 21" eq $name) {
      $name = "Galveston";
   }
   return $name;
}


#   Inputs to this subroutine are the four arrays which we read from the input
# .csv file: @rec_name, @rec_trend, @rec_lat, @rec_lon, and the number of
# records $num_recs.  Result is the geographically-weighted average MSL trend
# for the tide stations.
#   Also, if $debugmode>=1, then a lot of debug info is written to stdout.
#
#   For each station location, check it against every other location, calculate the
# distance D between the two locations, and the correlation weight W calculated
# from the distance.  Then (unless W=0) add the name/distance/weight triple
# to the list of nearby stations for that station location.
#   Also, for each location, sum the correlation weights of the nearby locations.
#   For each station, calculate its own weight W_own (to a maximum of 1.0) from the
# summed weights of the nearby stations, according to the formula:
#   W_own = 1.0 / (1.0 + sum(nearby_weights))
# Thus, an isolated station has a weight of 1.0, but a station near other stations
# has a lower weight; e.g., a station which is very near just 1 other station has a
# weight of about 0.5.
#   Finally, calculate the geographically weighted average linear MSL trend from all
# 159 stations, and return it as the subroutine result.
#
#   Also, there's an optional input parameter, which is the name of a file to
# which the W_own weights are to be written in the form of a .csv file, along
# with the key fields from the input file.
sub process_data {
   local( $outpfile ) = shift;
   local( $name1, $name2, $dist, $weight );
   if (defined $outpfile) {
      if (!open(OUTP,">$outpfile")) {
         printf "ERROR: could not create '%s', $!\n", $outpfile;
         exit 1;
      }
      # print a heading line for the output file:
      print OUTP '"Station_name",MSL_trend,"Latitude","Longitude",Weight' . "\n";
   }
   for ($s1 = 0; $s1 < $num_recs; $s1++) {
      $nearbyweight[$s1] = 0;
      $nearbylist[$s1] = '';
      $name1 = &city_name_only( $rec_name[$s1] );  # trim off country names, for conciseness
      for ($s2 = 0; $s2 < $num_recs; $s2++) {
         if ($s2 != $s1) {
            $name2 = &city_name_only( $rec_name[$s2] );
            $dist = &distance( $rec_lat[$s1], $rec_lon[$s1], $rec_lat[$s2], $rec_lon[$s2] ) / 1000.0;  # divided by 1000 to convert meters to km
            if ($dist < $km800) {
               $weight = &dist_to_weight( $dist );
               $nearbyweight[$s1] += $weight;
               $nearbylist[$s1] .= sprintf( "(%s, d=%0.1f, w=%0.3f)", $name2, $dist, $weight ) . "; ";
            }
         }
      }
      # this station has correlation weight 1.0; how much of the total is it for this location?
      $rec_weight[$s1] = 1.0 / (1.0 + $nearbyweight[$s1]);  # 1.0 for isolated tide stations, but as low as 0.07 for stations with many others nearby
      if ($debugmode) {
         # for each station, display the nearby station weight sum W_near, this station's weight W_own, and the list of nearby stations with their distances and correlation weights:
         printf "dbg: $s1. $name1, MSLt=%0.3f, lat/lon=%0.2f/%0.2f, W_near=%0.3f, W_own=%0.3f, list = %s\n",
                $rec_trend[$s1], $rec_lat[$s1], $rec_lon[$s1], $nearbyweight[$s1], $rec_weight[$s1], $nearbylist[$s1];
      }
      if (defined $outpfile) {
         print OUTP '"' . $rec_name[$s1] . '",' . $rec_trend[$s1] . ',"' . $rec_lat[$s1] . '","' . $rec_lon[$s1] . '",' . $rec_weight[$s1] . "\n";
      }
   }#for
   if (defined $outpfile) {
      close OUTP;
   }
   local($summed_weights) = 0.0;
   local($summed_weighted_trends) = 0.0;
   for ($s1 = 0; $s1 < $num_recs; $s1++) {
      $summed_weights += $rec_weight[$s1];
      $summed_weighted_trends += ($rec_weight[$s1] * $rec_trend[$s1]);
   }
   local($weighted_avg) = $summed_weighted_trends / $summed_weights;
   return $weighted_avg;
}

$avg_MSL_trend_geographically_weighted = &process_data( 'geo_weights.csv' );
print  "+---------------------------------------+\n";
printf "|   Weighted avg MSL trend = %0.06f   |\n", $avg_MSL_trend_geographically_weighted;
print  "+---------------------------------------+\n";
#
# Here's the result which this prints:
#  +---------------------------------------+
#  |   Weighted avg MSL trend = 1.133010   |
#  +---------------------------------------+
#


# Sanity/debug check:
#
# If this was calculated properly, then adding duplicate entries for any tide station
# will not much affect the result.
#
# In fact, it can't affect the result at all for isolated stations, like Guam (MSL
# trend -1.05 mm/yr), and Takoradi (MSL trend +3.35 mm/yr), which are more than 800 km
# from any other tide stations.
#
# For non-isolated stations (i.e., stations which are within 800 km of other stations),
# adding duplicate entries should have only a very small effect on the result.
#
# So, to sanity-check my code, here we examine the effect of adding duplicate station
# records, with four different stations: Guam, Takoradi, Furuogrund, and Galveston.
#
# As expected, the duplicate records do not affect the result at all for Guam or Takoradi,
# and affect it very little for Galveston and Furuogrund.  That gives confidence that
# the calculation was done correctly.
#

$true_num_recs = $num_recs;  # should be 159 (the number of GLOSS-LTT tide stations in NOAA's list)


# Add 100 duplicate Guam tide stations to the data set:
for ($num_recs = $true_num_recs; $num_recs < 259; $num_recs++) {
   $rec_trend[$num_recs] = $rec_trend[92];  # Guam (an isolated tide station w/ negative MSL trend)
   $rec_lat  [$num_recs] = $rec_lat  [92];
   $rec_lon  [$num_recs] = $rec_lon  [92];
   $rec_name [$num_recs] = &city_name_only($rec_name[92]) . ($num_recs + 1 - $true_num_recs);
}
$avg_MSL_trend_geographically_weighted = &process_data;
printf "Weighted avg MSL trend = %0.06f   (debug check with 100 extra Guams)\n", $avg_MSL_trend_geographically_weighted;
$num_recs = $true_num_recs + 10;
$avg_MSL_trend_geographically_weighted = &process_data;
printf "Weighted avg MSL trend = %0.06f   (debug check with 10 extra Guams)\n", $avg_MSL_trend_geographically_weighted;
$num_recs = $true_num_recs + 1;
$avg_MSL_trend_geographically_weighted = &process_data;
printf "Weighted avg MSL trend = %0.06f   (debug check with 1 extra Guam)\n", $avg_MSL_trend_geographically_weighted;


# Replace the duplicate Guam tide stations with duplicate Takoradi tide stations:
for ($num_recs = $true_num_recs; $num_recs < 259; $num_recs++) {
   $rec_trend[$num_recs] = $rec_trend[59];  # Takoradi (an isolated station w/ positive MSL trend)
   $rec_lat  [$num_recs] = $rec_lat  [59];
   $rec_lon  [$num_recs] = $rec_lon  [59];
   $rec_name [$num_recs] = &city_name_only($rec_name[59]) . ($num_recs + 1 - $true_num_recs);
}
$avg_MSL_trend_geographically_weighted = &process_data;
printf "Weighted avg MSL trend = %0.06f   (debug check with 100 extra Takoradis)\n", $avg_MSL_trend_geographically_weighted;
$num_recs = $true_num_recs + 10;
$avg_MSL_trend_geographically_weighted = &process_data;
printf "Weighted avg MSL trend = %0.06f   (debug check with 10 extra Takoradis)\n", $avg_MSL_trend_geographically_weighted;
$num_recs = $true_num_recs + 1;
$avg_MSL_trend_geographically_weighted = &process_data;
printf "Weighted avg MSL trend = %0.06f   (debug check with 1 extra Takoradi)\n", $avg_MSL_trend_geographically_weighted;


# Replace the dulicate stations with duplicate Furuogrund tide stations:
for ($num_recs = $true_num_recs; $num_recs < 259; $num_recs++) {
   $rec_trend[$num_recs] = $rec_trend[16];  # Furuogrund (has the most negative MSL trend of any tide station)
   $rec_lat  [$num_recs] = $rec_lat  [16];
   $rec_lon  [$num_recs] = $rec_lon  [16];
   $rec_name [$num_recs] = &city_name_only($rec_name[16]) . ($num_recs + 1 - $true_num_recs);
}
$avg_MSL_trend_geographically_weighted = &process_data;
printf "Weighted avg MSL trend = %0.06f   (debug check with 100 extra Furuogrunds)\n", $avg_MSL_trend_geographically_weighted;
$num_recs = $true_num_recs + 10;
$avg_MSL_trend_geographically_weighted = &process_data;
printf "Weighted avg MSL trend = %0.06f   (debug check with 10 extra Furuogrunds)\n", $avg_MSL_trend_geographically_weighted;
$num_recs = $true_num_recs + 1;
$avg_MSL_trend_geographically_weighted = &process_data;
printf "Weighted avg MSL trend = %0.06f   (debug check with 1 extra Furuogrund)\n", $avg_MSL_trend_geographically_weighted;


# Replace the dulicate stations with duplicate Galveston tide stations:
for ($num_recs = $true_num_recs; $num_recs < 259; $num_recs++) {
   $rec_trend[$num_recs] = $rec_trend[130];  # Galveston (has largest positive MSL trend of any tide station)
   $rec_lat  [$num_recs] = $rec_lat  [130];
   $rec_lon  [$num_recs] = $rec_lon  [130];
   $rec_name [$num_recs] = &city_name_only($rec_name[130]) . ($num_recs + 1 - $true_num_recs);
}
$avg_MSL_trend_geographically_weighted = &process_data;
printf "Weighted avg MSL trend = %0.06f   (debug check with 100 extra Galvestons)\n", $avg_MSL_trend_geographically_weighted;
$num_recs = $true_num_recs + 10;
$avg_MSL_trend_geographically_weighted = &process_data;
printf "Weighted avg MSL trend = %0.06f   (debug check with 10 extra Galvestons)\n", $avg_MSL_trend_geographically_weighted;
$num_recs = $true_num_recs + 1;
$avg_MSL_trend_geographically_weighted = &process_data;
printf "Weighted avg MSL trend = %0.06f   (debug check with 1 extra Galveston)\n", $avg_MSL_trend_geographically_weighted;


# But what if my aproximation function for the observed MSL Trend difference vs.
# distance wasn't quite right?  How much difference could that make?

# To answer that question, repeat the calculations with varying correlation
# curve approximations.  For simplicity, I again just use a three-segment
# piecewise-linear aproximating function, but vary the "knee" points.

# First, suppose that I've underestimated the extent to which nearby stations
# have correlated MSL trends.  Suppose the correlation doesn't drop to zero until
# tide stations are at least 1100 km apart (instead of 800 km):
$km400 = 550.00;
$km800 = 1100.00;
$num_recs = $true_num_recs;
$avg_MSL_trend_geographically_weighted = &process_data;
printf "Weighted avg MSL trend = %0.06f   (with correlation limit assumed to be 1100 km instead of 800 km)\n", $avg_MSL_trend_geographically_weighted;
# Here's the result:
#  Weighted avg MSL trend = 1.154084   (with correlation limit assumed to be 1100 km instead of 800 km)
# As you can see, there's very little difference.

# Next, suppose that I've overestimated the extent to which nearby stations
# have correlated MSL trends.  Suppose the correlation is actually zero for
# stations that are at least 600 km apart (instead of 800km ):
$km400 = 300.00;
$km800 = 600.00;
$num_recs = $true_num_recs;
$avg_MSL_trend_geographically_weighted = &process_data;
printf "Weighted avg MSL trend = %0.06f   (with correlation limit assumed to be 600 km instead of 800 km)\n", $avg_MSL_trend_geographically_weighted;
# Here's the result:
#  Weighted avg MSL trend = 1.102885   (with correlation limit assumed to be 600 km instead of 800 km)
# As you can see, again there's very little difference.

# Finally, for the sake of completeness, hypothesize and test just about every
# other possible maximum correlation distance:
print "\n";
print "--------------------------------------------------------------\n";
# Test basically all possible correlation distances:
$num_recs = $true_num_recs;
for ($km400 = 0; $km400 <= 10000;) {
   $km800 = 2 * $km400;
   $avg_MSL_trend_geographically_weighted = &process_data;
   printf "Weighted avg MSL trend = %0.06f  if correlation limit is assumed to be $km800 km", $avg_MSL_trend_geographically_weighted;
   if (800 != $km800) { print " (instead of 800 km)"; }
   print "\n";
   if ($km400 >= 4000) {
      $km400 += 500;
   } elsif ($km400 >= 2000) {
      $km400 += 250;
   } elsif ($km400 >= 600) {
      $km400 += 100;
   } else {
      $km400 += 50;
   }
}#for


exit 0;

__END__


# Here're the results of running this program:
#
#    C:\>perl calculate_distance_weighted_msl_avg.pl MSL_global_trendtable.csv
#    reading 'MSL_global_trendtable.csv' . . .
#    159 station records were read from MSL_global_trendtable.csv
#    *** Weighted avg MSL trend = 1.133010
#    Weighted avg MSL trend = 1.133010   (debug check with 100 extra Guams)
#    Weighted avg MSL trend = 1.133010   (debug check with 10 extra Guams)
#    Weighted avg MSL trend = 1.133010   (debug check with 1 extra Guam)
#    Weighted avg MSL trend = 1.133010   (debug check with 100 extra Takoradis)
#    Weighted avg MSL trend = 1.133010   (debug check with 10 extra Takoradis)
#    Weighted avg MSL trend = 1.133010   (debug check with 1 extra Takoradi)
#    Weighted avg MSL trend = 1.153640   (debug check with 100 extra Furuogrunds)
#    Weighted avg MSL trend = 1.127484   (debug check with 10 extra Furuogrunds)
#    Weighted avg MSL trend = 1.130163   (debug check with 1 extra Furuogrund)
#    Weighted avg MSL trend = 1.130410   (debug check with 100 extra Galvestons)
#    Weighted avg MSL trend = 1.134232   (debug check with 10 extra Galvestons)
#    Weighted avg MSL trend = 1.134416   (debug check with 1 extra Galveston)
#    Weighted avg MSL trend = 1.154084   (with correlation limit assumed to be 1100 km instead of 800 km)
#    Weighted avg MSL trend = 1.102885   (with correlation limit assumed to be 600 km instead of 800 km)
#    --------------------------------------------------------------
#    Weighted avg MSL trend = 0.611132  if correlation limit is assumed to be 0 km (instead of 800 km)
#    Weighted avg MSL trend = 0.590453  if correlation limit is assumed to be 100 km (instead of 800 km)
#    Weighted avg MSL trend = 0.803731  if correlation limit is assumed to be 200 km (instead of 800 km)
#    Weighted avg MSL trend = 0.924700  if correlation limit is assumed to be 300 km (instead of 800 km)
#    Weighted avg MSL trend = 1.006062  if correlation limit is assumed to be 400 km (instead of 800 km)
#    Weighted avg MSL trend = 1.064766  if correlation limit is assumed to be 500 km (instead of 800 km)
#    Weighted avg MSL trend = 1.102885  if correlation limit is assumed to be 600 km (instead of 800 km)
#    Weighted avg MSL trend = 1.128633  if correlation limit is assumed to be 700 km (instead of 800 km)
#    Weighted avg MSL trend = 1.133010  if correlation limit is assumed to be 800 km
#    Weighted avg MSL trend = 1.137817  if correlation limit is assumed to be 900 km (instead of 800 km)
#    Weighted avg MSL trend = 1.145390  if correlation limit is assumed to be 1000 km (instead of 800 km)
#    Weighted avg MSL trend = 1.154084  if correlation limit is assumed to be 1100 km (instead of 800 km)
#    Weighted avg MSL trend = 1.164056  if correlation limit is assumed to be 1200 km (instead of 800 km)
#    Weighted avg MSL trend = 1.174616  if correlation limit is assumed to be 1400 km (instead of 800 km)
#    Weighted avg MSL trend = 1.163980  if correlation limit is assumed to be 1600 km (instead of 800 km)
#    Weighted avg MSL trend = 1.162185  if correlation limit is assumed to be 1800 km (instead of 800 km)
#    Weighted avg MSL trend = 1.163626  if correlation limit is assumed to be 2000 km (instead of 800 km)
#    Weighted avg MSL trend = 1.169893  if correlation limit is assumed to be 2200 km (instead of 800 km)
#    Weighted avg MSL trend = 1.174270  if correlation limit is assumed to be 2400 km (instead of 800 km)
#    Weighted avg MSL trend = 1.182248  if correlation limit is assumed to be 2600 km (instead of 800 km)
#    Weighted avg MSL trend = 1.195531  if correlation limit is assumed to be 2800 km (instead of 800 km)
#    Weighted avg MSL trend = 1.207766  if correlation limit is assumed to be 3000 km (instead of 800 km)
#    Weighted avg MSL trend = 1.216136  if correlation limit is assumed to be 3200 km (instead of 800 km)
#    Weighted avg MSL trend = 1.219038  if correlation limit is assumed to be 3400 km (instead of 800 km)
#    Weighted avg MSL trend = 1.215959  if correlation limit is assumed to be 3600 km (instead of 800 km)
#    Weighted avg MSL trend = 1.212590  if correlation limit is assumed to be 3800 km (instead of 800 km)
#    Weighted avg MSL trend = 1.214074  if correlation limit is assumed to be 4000 km (instead of 800 km)
#    Weighted avg MSL trend = 1.207411  if correlation limit is assumed to be 4500 km (instead of 800 km)
#    Weighted avg MSL trend = 1.189336  if correlation limit is assumed to be 5000 km (instead of 800 km)
#    Weighted avg MSL trend = 1.174343  if correlation limit is assumed to be 5500 km (instead of 800 km)
#    Weighted avg MSL trend = 1.162298  if correlation limit is assumed to be 6000 km (instead of 800 km)
#    Weighted avg MSL trend = 1.147830  if correlation limit is assumed to be 6500 km (instead of 800 km)
#    Weighted avg MSL trend = 1.137204  if correlation limit is assumed to be 7000 km (instead of 800 km)
#    Weighted avg MSL trend = 1.130473  if correlation limit is assumed to be 7500 km (instead of 800 km)
#    Weighted avg MSL trend = 1.121765  if correlation limit is assumed to be 8000 km (instead of 800 km)
#    Weighted avg MSL trend = 1.096313  if correlation limit is assumed to be 9000 km (instead of 800 km)
#    Weighted avg MSL trend = 1.068146  if correlation limit is assumed to be 10000 km (instead of 800 km)
#    Weighted avg MSL trend = 1.030950  if correlation limit is assumed to be 11000 km (instead of 800 km)
#    Weighted avg MSL trend = 0.990063  if correlation limit is assumed to be 12000 km (instead of 800 km)
#    Weighted avg MSL trend = 0.953041  if correlation limit is assumed to be 13000 km (instead of 800 km)
#    Weighted avg MSL trend = 0.922263  if correlation limit is assumed to be 14000 km (instead of 800 km)
#    Weighted avg MSL trend = 0.894711  if correlation limit is assumed to be 15000 km (instead of 800 km)
#    Weighted avg MSL trend = 0.870387  if correlation limit is assumed to be 16000 km (instead of 800 km)
#    Weighted avg MSL trend = 0.849261  if correlation limit is assumed to be 17000 km (instead of 800 km)
#    Weighted avg MSL trend = 0.830635  if correlation limit is assumed to be 18000 km (instead of 800 km)
#    Weighted avg MSL trend = 0.813638  if correlation limit is assumed to be 19000 km (instead of 800 km)
#    Weighted avg MSL trend = 0.798965  if correlation limit is assumed to be 20000 km (instead of 800 km)
#
# As you can see, for no possible maximum correlation distance does the weighted
# global average linear mean sea level rise exceed aprox. 1.2 mm/year.

