#!/usr/bin/perl

# Run this program with no parameters for help.
#
# (Written for Perl 5.008, but compatible even with Perl 4.0036 if you delete
# the "use bignum" directive.)
#
# Copyright 2010-2012, by David A. Burton
# Cary, NC  USA
# +1 919-481-0149
# Email: http://www.burtonsys.com/email/
#

# TLIB Version Control fills in the version information for us:
$version_str = "";
#--=>keyflag<=-- "&(#)%n, version %v, %d "
$version_str = "&(#)church_white_regressions.pl, version 18, 05-Feb-12 ";


# immediate output of debug prints
$| = 1;


$debugmode=0;  # for debug prints

$opt_firstyear = 1000;  # '-f=1000' is the default (distant past)
$opt_lastyear = 3000;  # '-l=3000' is the default (distant future)
$opt_quiet = 0;      # '-q' (quiet) says to NOT view the result before exiting
$opt_outputfile = 'church_white_regressions.bmp';
$opt_compress = 0;  # 1 to compress output with Irfanview (Windows only)
$opt_weights = 1;  # 1 = equal weights, 2 = reciprocal of variance, 4 = reciprocal of standard deviation
$opt_width = 700;  # default width of .bmp file, in pixels
$opt_height = 500;  # default height of .bmp file, in pixels
$opt_caption = '';  # for adding a title line to the graph
$opt_copr = '';  # for added a summary or copyright line to the graph

$weight_equal_c = 1;
$weight_variance_c = 2;
$weight_SD_c = 4;
@weight_name = ('none', 'equal_weights', '1/variance', 'equal_weights & 1/variance',
                '1/standard_deviation', 'equal_weights & 1/standard_deviation' );


print "perl $0 " . join(' ',@ARGV) . "\n";


while (($#ARGV >= 0) && ('-' eq substr($ARGV[0],0,1))) {
   if ($ARGV[0] =~ /^\-d$/i) {
      $debugmode++;  # turn on debug prints
   } elsif ($ARGV[0] =~ /^\-c$/i) {
      $opt_compress++;
   } elsif ('-q' eq $ARGV[0]) {
      $opt_quiet++;  # disable automatic viewing of resulting .bmp file
   } elsif ($ARGV[0] =~ /^\-h\=([0-9]+)$/i) {
      $tmp = $1 + 0;
      if (($tmp < 100) || ($tmp > 4000)) {
         print "ERROR: '$ARGV[0]'  output file must be 100..4000 pixels high\n";
      } else {
         $opt_height = $tmp;
      }
   } elsif ($ARGV[0] =~ /^\-w\=([0-9]+)$/i) {
      $tmp = $1 + 0;
      if (($tmp < 100) || ($tmp > 5000)) {
         print "ERROR: '$ARGV[0]'  output file must be 100..5000 pixels wide\n";
      } else {
         $opt_width = $tmp;
      }
   } elsif ($ARGV[0] =~ /^\-f\=([12][0-9]{3}(\.[0-9]*|))$/i) {
      $tmp = $1 + 0;
      $opt_firstyear = $tmp;
      if ($debugmode) {
         print "'" . $ARGV[0] . "', Limit data to first year = $opt_firstyear \& later\n";
      }
   } elsif ($ARGV[0] =~ /^\-a\=([12345])$/) {
      $opt_weights = $1 + 0;
      if ($debugmode) {
         print "'" . $ARGV[0] . "', Weighting factor = " . $weight_name[$opt_weights] . "\n";
      }
   } elsif ($ARGV[0] =~ /^\-A\=(.*)$/) {
      $opt_caption = $1;
      if ($debugmode) {
         print "caption = '$opt_caption'\n";
      }
   } elsif ($ARGV[0] =~ /^\-Z\=(.*)$/) {
      $opt_copr = $1;
      if ($debugmode) {
         print "copr/summary = '$opt_copr'\n";
      }
   } elsif ($ARGV[0] =~ /^\-o\=([0-9a-z\_\-\$\.\\\/\:\(\)]*)$/i) {
      $opt_outputfile = $1;
      if (($opt_outputfile ne '') && ($opt_outputfile !~ /\.[^\:\\\/]*$/)) {
         # if there's no filename extension, add ".bmp"
         $opt_outputfile .= '.bmp';
      }
      if ($debugmode) {
         print "'" . $ARGV[0] . "', Set output file to '$opt_outputfile'\n";
      }
   } elsif ($ARGV[0] =~ /^\-l\=([12][0-9]{3}(\.[0-9]*|))$/i) {
      $tmp = $1 + 0;
      $opt_lastyear = $tmp;
      if ($debugmode) {
         print "'" . $ARGV[0] . "', Limit data to last year = $opt_lastyear \& earlier\n";
      }
   } else {
      printf "ERROR: unrecognized command-line option: '%s'\n", $ARGV[0];
      exit 1;
   }
   shift @ARGV;
}


$perlver = "3 or earlier";
if ($] =~ /\$\$Revision\:\s*([0-9.]+)\s/) {
   $perlver = $1;  # probably 4.something
} elsif ($] =~ /([0-9][0-9.]*)/) {
   $perlver = $1;  # probably 5.something or 6.something
}
if ($debugmode) {
   print "You are using Perl version $perlver\n";
}


# make sure we fetch these modules from the current folder or (2nd choice) the parent folder
unshift( @INC, '..' );
unshift( @INC, '.' );
require "plot_to_bmp.pl";
use bignum;
require "linear_fit.pl";
require "quadratic_fit.pl";
require "weighted_linear_fit.pl";
require "weighted_quadratic_fit.pl";
no bignum;
shift( @INC );  # restore
shift( @INC );



$num_args = $#ARGV+1;

if ($num_args != 1) {
 print "church_white_regressions.pl -- read Church and White monthly or yearly data,\n" .
       "and do regression analysis (least-squares fit) to line and quadratic.\n" .
       "(Input files are from  http://www.pol.ac.uk/psmsl/author_archive/church_white/)\n" .
       "\n" .
       "Usage:\n" .
       "\n" .
       "   perl $0 [options] church_white_grl_gmsl.lis\n" .
       "\n" .
       "where [options] can be a combination (or none) of the following:\n" .
       "   -f=xxxx  Ignore input data with dates < year xxxx\n" .
       "   -L=yyyy  Ignore input data with dates > year yyyy\n" .
       "   -o=filename  Output to filename (instead of 'church_white_regressions.bmp')\n" .
       "       filename should end in .bmp (or .png if you also specify '-c' to compress)\n" .
       "   -a=n  where n is the adjusted weighting factor:\n" .
       "           1 = equal weights (default),  2 = 1/variance,  4 = 1/SD  (or 1+2=3, or 1+4=5)\n" .
       "   -A=\"Caption\"  annotate graph with specified caption\n" .
       "   -c  compress output to 1/6 size with Irfanview, or to 1/100 if output filename\n" .
       "       ends in .png (Windows only!)\n" .
       "   -q  (quiet) to NOT view the result before exiting\n" .
       "   -d  enable Debug prints\n" .
       "   -w=xxx  create output file xxx pixels wide (default 700)\n" .
       "   -h=xxx  create output file xxx pixels high (default 500)\n" .
       "\n";
       exit 1;
}

$inpfile = $ARGV[0];

if (!open(INP,"<$inpfile")) {
   printf "ERROR: could not open '%s', $!\n", $inpfile;
   exit 1;
}
if ($debugmode) {
   print "reading '$inpfile'...\n";
}


# put commas in an integer if it is > 4 digits long
sub commafy {
   local($number) = shift;
   local(@pieces) = ();
   $number .= '';
   if (length($number) > 4) {
      while (length($number) > 0) {
         unshift(@pieces,substr($number,-3));
         substr($number,-3) = '';
      }
      $number = join(',',@pieces);
   }
   return $number;
}



# SAMPLE MONTHLY INPUT:
#
#   years    GMSL (mm)   SD
# ---------  ---------  -----
# 1870.0417   -75.61    22.19
# 1870.1250   -75.68    22.19
# 1870.2083   -74.13    22.19
# 1870.2917   -74.06    22.19
# 1870.3750   -73.23    22.19
# 1870.4583   -74.47    22.19
# 1870.5417   -72.76    22.19
# 1870.6250   -70.71    22.19
# 1870.7083   -71.18    22.19
# 1870.7917   -70.77    22.19
# 1870.8750   -78.15    22.19
# 1870.9583   -81.73    22.19
# 1871.0417   -86.35    22.19
# 1871.1250   -82.06    22.19
# 1871.2083   -78.42    22.19
#...
# 2001.7917   118.71     8.94
# 2001.8750   114.17     9.05
# 2001.9583   110.85     9.05


# SAMPLE YEARLY INPUT:
#
#  years   GMSL (mm)   SD
# -------  ---------  -----
#  1870.5  -104.53    22.73
#  1871.5  -108.49    22.73
#...
#  2001.5   115.24     5.07
#  2002.5   115.28     5.17
#  2003.5   121.73     5.38
#  2004.5   123.41     6.11
#  2005.5   128.05     6.43
#  2006.5   141.67     6.87
#  2007.5   137.64     7.22


# Pass in two arrays, @x and @y, of identical length; get back an array
# of x,y pairs.
sub pairwise {
   local( $numpts ) = ((1+$#_) / 2);
   local( @x ) = splice( @_, 0, $numpts );
   local( @y ) = @_;
   local( @xy );
   local( $i );
   for ($i=0; $i<$numpts; $i++) {
      push( @xy, $x[$i], $y[$i] );
   }
   return @xy;
}


# Pass in an array of x,y pairs; get back just the x values
sub xvalues {
   local( @x );
   while ($#_ >= 0) {
      push( @x, (shift @_) );
      shift @_;
   }
   return @x;
}


# Pass in an array of x,y pairs; get back just the x values
sub yvalues {
   local( @y );
   while ($#_ > 0) {
      shift @_;
      push( @y, (shift @_) );
   }
   return @y;
}


$lightgrey_c = &color24(0xE0DFD0);
# $very_light_yellow_c = &color24(0xFFFAD2);


# Plot a two-toned vertical line segment representing the 1-SD and 2-SD CIs for a data point
sub plot_CI {
   local( $x, $y, $sd, $dotsize ) = @_;
   local( $i, $ymin2, $ymin1, $ymax1, $ymax2, $incr, @xy_ci1, @xy_ci2 );
   $ymin1 = $y - $sd;
   $ymin2 = $ymin1 - $sd;
   $ymax1 = $y + $sd;
   $ymax2 = $ymax1 + $sd;
   $incr = $scaleY / 3;
   for ($i = $ymin2; $i < $ymin1; $i += $incr) {
      push( @xy_ci2, $x );
      push( @xy_ci2, $i );
   }
   for ($i = $ymax2; $i > $ymax1; $i -= $incr) {
      push( @xy_ci2, $x );
      push( @xy_ci2, $i );
   }
   for ($i = $ymin1; $i <= $ymax1; $i += $incr) {
      push( @xy_ci1, $x );
      push( @xy_ci1, $i );
   }
   push( @xy_ci1, $x );
   push( @xy_ci1, $ymax1 );
   # plot the 2-SD (~95%) confidence interval in light grey
   plot_points2( $lightgrey_c, $dotsize, @xy_ci2 );
   # plot the 1-SD interval in medium grey
   plot_points2( $grey_c, $dotsize, @xy_ci1 );
}



# Read the input file, loading up
@xy = @x = @y = @sd = ();
$count_inp_lines = 0;
$warned_firstyear = 0;
$year_i = $gmsl_i = $sd_i = 0;
while (<INP>) {
   $count_inp_lines++;
   $_ =~ s/^\s+//g;  # delete any leading whitespace
   ($year_i, $gmsl_i, $sd_i) = split( /\s+/, $_ );
   if ((1==$count_inp_lines) && ($year_i =~ /year/i)) {
      next;  # if there's a header line, ignore it
   }
   if ((0+$year_i) < $opt_firstyear) {
      if (! $warned_firstyear) {
         print "*** Ignoring data before $opt_firstyear ***\n";
         $warned_firstyear = 1;
      }
      next;
   }
   if ((0+$year_i) > $opt_lastyear) {
      print "*** Ignoring data after $opt_lastyear ***\n";
      last;
   }
   push( @x, $year_i );
   push( @y, $gmsl_i );
   push( @sd, $sd_i );
}
close INP;
print "$0 read $count_inp_lines lines from '$inpfile'\n";

if ($debugmode) {
   print "dbg: -------- dump xy:\n";
   for ($i=0; $i <= $#x; $i++) {
      print "dbg: i=$i, xy[$i]=($x[$i],$y[$i])\n";
   }
   print "dbg: -------- dump of xy is complete.\n";
}

use bignum;

# If not using bignum, you'll get more reliable results by biasing the x-axis values close to zero
# foreach $yr (@x) {
#    $yr -= 1984;
# }

# Least-squares fit of a line to the data:
@xy = &pairwise( @x, @y );
($m, $b, $r) = &linear_fit( @xy );
print "Linear regression:\n";
print "  Slope = $m,  Y-intercept = $b,  R=$r\n";

# Least-squares fit of a quadratic to the data:
print "Quadratic fit:\n";
($A, $B, $C, $A2, $B2, $C2) = &quadratic_fit( @xy );
printf "  Y  =  (A=%f)x^2  +  (B=%f)x  +  (C=%f)\n", $A, $B, $C;
# (A,B,C) is calculated by my primary method.  (A2,B2,C2) are calculated by code
# adapted from pfit by John Lapeyre.  They should be the same.
if (!defined $C2) {
   # one of these days I'm going to delete the 2nd method from quadratic_fit.pl
   $A2 = $A;
   $B2 = $B;
   $C2 = $C;
}
printf "  Y  =  (A=%f)x^2  +  (B=%f)x  +  (C=%f)\n", $A2, $B2, $C2;

# For weighted least-squares fits of line or curve to the data, we first need
# to calculate the weights.
# http://en.wikipedia.org/wiki/Least_squares#Weighted_least_squares  says:
#   "Aitken showed that when a weighted sum of squared residuals is minimized,
#   Beta is a best linear unbiased estimator if each weight is equal to the
#   reciprocal of the variance of the measurement."
# So the "right" weighting is supposedly the reciprocals of the variances.
# That's "-a=2".  But I also allow "-a=1" (equal weights) and "-a=4" (1/SD).
@w = @sd;
for ($i=0; $i<=$#sd; $i++) {
   $w[$i] = 1;
   if (0!=$sd[$i]) {
      if (0 != ($opt_weights & $weight_variance_c)) {
         $w[$i] = 1 / ($sd[$i] * $sd[$i]);
      } elsif (0 != ($opt_weights & $weight_SD_c)) {
         $w[$i] = 1 / $sd[$i];
      }
   }
}
($m2, $b2) = &weighted_linear_fit( @x, @y, @w );
print "Weighted linear regression:\n";
print "  Slope = $m2,  Y-intercept = $b2\n";

# Weighted least-squares fit of a quadratic curve to the data.
($A3, $B3, $C3, $A4, $B4, $C4) = &weighted_quadratic_fit( @x, @y, @w );
print "Weighted quadratic fit:\n";
printf "  Y  =  (A=%f)x^2  +  (B=%f)x  +  (C=%f)\n", $A3, $B3, $C3;
printf "  Y  =  (A=%f)x^2  +  (B=%f)x  +  (C=%f)\n", $A4, $B4, $C4;

# Now, calculate the projected 21st century sea level rise:
$interpolated_2000_level = ($A * (2000*2000)) + ($B * 2000) + $C;
$extrapolated_2100_level = ($A * (2100*2100)) + ($B * 2100) + $C;
$extrapolated_2100_rise = $extrapolated_2100_level - $interpolated_2000_level;
$quadratic_projection = sprintf( "21st century quadratic-projected rise = %.1f mm", $extrapolated_2100_rise );
print "$quadratic_projection\n";

$linear_projection = sprintf( "21st century linear-projected rise = %.1f mm", 100*$m );
print "$linear_projection\n";

$weighting = 'non-';
if (0 != ($opt_weights & $weight_variance_c)) {
   $weighting = '1/V ';
} elsif (0 != ($opt_weights & $weight_SD_c)) {
   $weighting = '1/SD ';
}
$interpolated_2000_level_wq = ($A3 * (2000*2000)) + ($B3 * 2000) + $C3;
$extrapolated_2011_level_wq = ($A3 * (2011*2011)) + ($B3 * 2011) + $C3;
$extrapolated_2080_level_wq = ($A3 * (2080*2080)) + ($B3 * 2080) + $C3;
$extrapolated_2100_level_wq = ($A3 * (2100*2100)) + ($B3 * 2100) + $C3;
$extrapolated_2100_rise_wq = $extrapolated_2100_level_wq - $interpolated_2000_level_wq;
$extrapolated_69yr_rise_wq = $extrapolated_2080_level_wq - $extrapolated_2011_level_wq;
$extrapolated_89yr_rise_wq = $extrapolated_2100_level_wq - $extrapolated_2011_level_wq;
$weighted_quadratic_projection = sprintf( "21st century ${weighting}weighted-quadratic-projected rise = %.1f mm", $extrapolated_2100_rise_wq );
$weighted_quadratic_69yr_projection = sprintf( "69 year (2011-2080) ${weighting}weighted-quadratic-projected rise = %.1f mm", $extrapolated_69yr_rise_wq );
$weighted_quadratic_89yr_projection = sprintf( "89 year (2011-2100) ${weighting}weighted-quadratic-projected rise = %.1f mm", $extrapolated_89yr_rise_wq );
print "$weighted_quadratic_projection\n";
print "$weighted_quadratic_69yr_projection\n";
print "$weighted_quadratic_89yr_projection\n";

$weighted_linear_projection = sprintf( "21st century ${weighting}weighted-linear-projected rise = %.1f mm", 100*$m2 );
$weighted_69yr_linear_projection = sprintf( "69 year (2011-2080) ${weighting}weighted-linear-projected rise = %.1f mm", 69*$m2 );
$weighted_89yr_linear_projection = sprintf( "89 year (2011-2100) ${weighting}weighted-linear-projected rise = %.1f mm", 89*$m2 );
print "$weighted_linear_projection\n";
print "$weighted_69yr_linear_projection\n";
print "$weighted_89yr_linear_projection\n";


no bignum;


# Initialize the graph.  I like a light yellow background for the graph.
# Must call this before calling the calc_adjusted_* or plot_* subroutines.
&initialize_bmp( $opt_width, $opt_height, $very_light_yellow_c );


# This is for setting the scales of the graphs.  We do it here because we need
# to at least determine the range of X-axis values (years) before calling
# gen_function_plot, since it needs to vary X over that range.  Also, since
# we're going to want to project the trend lines to 2100, I added the "2100"
# year, to ensure that the X-axis will extend that far:
&find_min_and_max_X( @x,2102 );
&find_min_and_max_Y( @y );
# For the Y-axis, we need to make room to plot the 1-SD and 2-SD confidence intervals
@ytmp1 = @ytmp2 = @y;
for ($i=0; $i<=$#y; $i++) {
   $ytmp1[$i] = $y[$i] - (2 * $sd[$i]);
   $ytmp2[$i] = $y[$i] + (2 * $sd[$i]);
}
&find_min_and_max_Y( @ytmp1, @ytmp2 );
&calc_adjusted_X_range;

@xy1 = @xy2 = @xy3 = @xy4 = @xy5 = @xy6 = ();

if (0 != ($opt_weights & $weight_equal_c)) {
   # Calculate the linear trend line X,Y plot values:
   $linfunc = "(($m * \$x) + $b)";
   if ($debugmode) {
      print "linfunc = '$linfunc'\n";
   }
   @xy1 = &gen_function_plot( $minX, $maxX, $linfunc );

   # Calculate the quadratic curve X,Y plot values:
   $quadfunc1 = "(($A * \$x**2) + ($B * \$x) + $C)";
   $quadfunc2 = "(($A2 * \$x**2) + ($B2 * \$x) + $C2)";  # John Lapeyre method
   if ($debugmode) {
      print "quadfunc1 = '$quadfunc1'\n";
   }
   @xy2 = &gen_function_plot( $minX, $maxX, $quadfunc1 );
   @xy3 = &gen_function_plot( $minX, $maxX, $quadfunc2 );
   # @xy3 should be identical to @xy2 (and, if you use bigint, it really is identical).
}

if (0 != ($opt_weights & ($weight_variance_c|$weight_SD_c))) {
   # Calculate the weighted-linear trend line X,Y plot values:
   $linfunc2 = "(($m2 * \$x) + $b2)";
   if ($debugmode) {
      print "linfunc2 = '$linfunc2'\n";
   }
   @xy4 = &gen_function_plot( $minX, $maxX, $linfunc2 );

   $quadfunc3 = "(($A3 * \$x**2) + ($B3 * \$x) + $C3)";
   $quadfunc4 = "(($A4 * \$x**2) + ($B4 * \$x) + $C4)";
   if ($debugmode) {
      print "weighted-quadfunc = '$quadfunc3'\n";
   }
   @xy5 = &gen_function_plot( $minX, $maxX, $quadfunc3 );
   @xy6 = &gen_function_plot( $minX, $maxX, $quadfunc4 );
   # @xy6 should be identical to @xy5 (and, if you use bigint, it really is identical).
}


# Before plotting the axes and adding the ticks & labels, readjust the scales,
# since the plotted curves might extend outside the original data's Y range:
&find_min_and_max_XY( @xy, @xy1, @xy1, @xy2, @xy3, @xy4, @xy5, @xy6 );
&find_min_and_max_X( $x[0]-9, 2107 );  # make a little more room at the left and right, so the "2100" label will display
&calc_adjusted_XY_ranges;


# Plot the confidence interval indicators for each X,Y point.  We plot them first
# (before even the axes) so that they don't cover up more important things.
for ($i = 0; $i <= $#y; $i++) {
   if ($#x < 500) {
      $dotsize = 3;  # for yearly data, we plot with 3-pixel-wide dots
   } else {
      $dotsize = 1;  # for monthly data, we plot with 1-pixel-wide dots
   }
   &plot_CI( $x[$i], $y[$i], $sd[$i], $dotsize );
}

# Plot the points and curves on the graph.  For prettiest result, I first plot
# with big dots and lines, them I plot over that with thin ones.  The goal is to
# ensure that all the dots are visible, but they won't obscure the trend curves.

if ($#x < 500) {
   # for monthly data, don't plot the really big (9x9) dots
   &plot_points2( $darkgrey_c, 9, @xy );  # big data points
}
&plot_points2( $lightgreen_c, 5, @xy1 );  # thick linear
&plot_points2( $carolina_blue_c, 5, @xy4 );  # thick weighted-linear
&plot_points2( $red_c, 5, @xy3 );  # thick quadratic
&plot_points2( $orange_c, 5, @xy5 );  # thick weighted-quadratic
&plot_points2( $darkgrey_c, 5, @xy );  # medium data points

# Draw the axes on the graph:
&plot_axes;

# Draw the grid lines and axis labels on the graph:
&add_ticks(1);

&plot_points2( $green_c, 1, @xy1 );  # thin linear
&plot_points2( $blue_c, 1, @xy4 );  # thin weighted-linear
&plot_points2( $purple_c, 1, @xy2 );  # thin quadratic
&plot_points2( $brown_c, 1, @xy6 );  # thin weighted-quadratic
&plot_points2( $black_c, 1, @xy );  # tiny data points


# $red_c = &color24(0xFF0000);
# $purple_c = &color24(0x8000FF);
$purpleish_red_c = &color24(0xBF007F);
$dark_orange_c = &color24(0xC85000);


# This is a string representation of the linear function which we'll draw onto the graph, in green (to match the green trend line):
$linfuncstr1 = 'Y = (' . sprintf("%1.4f",$m) . " * X) + " . sprintf("%4.1f",$b);

# This is a string representation of the quadratic function which we'll draw onto the graph, in red (to match the red/purple trend line):
$quadfuncstr1 = 'Y = (' . sprintf("%1.7f",$A) . " * X\xFD) + (" . sprintf("%1.4f",$B) . " * X) + " .sprintf("%4.1f",$C);

# This is a string representation of the weighted-linear function which we'll draw onto the graph, in blue (to match the blue trend line):
$linfuncstr2 = 'Y = (' . sprintf("%1.4f",$m2) . " * X) + " . sprintf("%4.1f",$b2);

# This is a string representation of the weighted-quadratic function which we'll draw onto the graph, in orange (to match the orange trend line):
$quadfuncstr2 = 'Y = (' . sprintf("%1.7f",$A3) . " * X\xFD) + (" . sprintf("%1.4f",$B3) . " * X) + " .sprintf("%4.1f",$C3);

# Default graph is 500 pixels high, so (60, 450) is 60 pixels from the left margin & 50 pixels from the top
$txt_Y = $opt_height-20;
if ('' ne $opt_caption) {
   &drawstr( 60, $txt_Y, $opt_caption, $black_c );  # show the caption, if any
   $txt_Y -= 16;
}
&drawstr( 60, $txt_Y, "\xDB", $grey_c );
   &drawstr( 67, $txt_Y, "\xDB", $lightgrey_c );
   &drawstr( 74, $txt_Y, " Dark \& light grey represent 1xSD \& 2xSD confidence intervals, respectively", $darkgrey_c );
   $txt_Y -= 16;
$txt_Y -= 2;
if (0 != ($opt_weights & $weight_equal_c)) {
   &drawstr( 60, $txt_Y, $quadratic_projection, $purpleish_red_c  );  # print the quadratic projection
   $txt_Y -= 12;
   &drawstr( 60, $txt_Y, $quadfuncstr1, $purpleish_red_c  );  # print the quadratic function
   $txt_Y -= 18;
   &drawstr( 60, $txt_Y, $linear_projection, $green_c );  # print the linear projection
   $txt_Y -= 12;
   &drawstr( 60, $txt_Y, $linfuncstr1, $green_c );  # print the linear function
   $txt_Y -= 18;
}
if (0 != ($opt_weights & ($weight_variance_c|$weight_SD_c))) {
   &drawstr( 60, $txt_Y, $weighted_quadratic_projection, $dark_orange_c );  # print the weighted-quadratic projection
   $txt_Y -= 12;
   &drawstr( 60, $txt_Y, $quadfuncstr2, $dark_orange_c );  # print the weighted-quadratic function
   $txt_Y -= 18;
   &drawstr( 60, $txt_Y, $weighted_linear_projection, $blue_c );  # print the weighted-linear projection
   $txt_Y -= 12;
   &drawstr( 60, $txt_Y, $linfuncstr2, $blue_c );  # print the weighted-linear function
   $txt_Y -= 18;
}
if ('' ne $opt_copr) {
   &drawstr( 60, $txt_Y, $opt_copr, $black_c );  # show the copyright or summary line, if any
   $txt_Y -= 16;
}


# write the graph to $opt_outputfile

if ($opt_compress) {
   # if '-c' specified, then compress it with IrfanView
   unlink( $opt_outputfile );
   &write_bmp( 'church_white_regressions_tmp.bmp' );
   $cmd = '"%ProgramFiles%\\IrfanView\\i_view32.exe" church_white_regressions_tmp.bmp /bpp=4 /transpcolor= /convert=' . $opt_outputfile;
   print "Compressing output...\n$cmd\n";
   `$cmd`;
   if (-f $opt_outputfile) {
      print "IrfanView compressed $opt_outputfile from " . &commafy(-s 'church_white_regressions_tmp.bmp') .
            " bytes to " . &commafy(-s $opt_outputfile) . " bytes.\n";
      unlink( 'church_white_regressions_tmp.bmp' );
   } else {
      warn "ERROR: compression via IrfanView failed; perhaps it isn't installed.\n";
      rename( 'church_white_regressions_tmp.bmp', $opt_outputfile );
   }
} else {
   &write_bmp( $opt_outputfile );
}

$fsize = -s $opt_outputfile;
print "Created plot: $opt_outputfile (" . &commafy($fsize) . " bytes)\n";

# inspect it it, using whatever the default viewer is for .bmp files (unless user specified '-q'):
if (!$opt_quiet) {
   print "About to tell OS to open '$opt_outputfile' . . .\n";
   `$opt_outputfile`;
   print "Back from opening '$opt_outputfile'\n";
}
print "\n";  # a blank line at the end is nice when doing several runs in a batch script

exit 0;

__END__

