#!/usr/bin/perl

# Function to determine whether its caller was invoked via 'do'/'require'
# or via the command line.
#
# (Compatible with both Perl4 and Perl5.)
#
# This file is UNCOPYRIGHTED, by David A. Burton, 2010
# Cary, NC  USA
# +1-919-481-0149
# Email: http://www.burtonsys.com/email/
#
# You can use a "require" to pull this into another Perl program, and use it like this:
#
#   @INC = ('.','..');
#   require "detect_do_or_require.pl";
#      ...
#   if (&invoked_via_do_or_require) {
#      print "$0 was loaded via 'do', 'require', or 'use'\n";
#   } else {
#      print "$0 was invoked from command-line\n";
#   }
#


# require 'traceback.pl';

# If this returns 1, then the caller's Perl source file was loaded via a 'do' or 'require' or 'use'.
# If this returns 0, then the caller's Perl source file was loaded from the command line.
sub invoked_via_do_or_require {
   local($was_invoked_via_do_or_require);
   local($package, $filename) = caller(1);
   # In Perl 5, $filename is the Perl source file name for the caller of the caller of this function
   # &traceback( 'at top of sub invoked_via_do_or_require' );
   if (defined $filename) {
      # in Perl 5, if the caller's source file was DOne/REQUIREd then there's an 'eval' stack frame before this subroutine's stack frame
      $was_invoked_via_do_or_require = 1;
      # print "dbg: caller(1) filename is defined ($filename), so must be Perl 5 DO/REQUIRE\n";
   } else {
      # if there's no 2nd caller frame then either this is Perl 4 or else the caller's source file wasn't DOne/REQUIREd
      ($package, $filename) = caller(0);
      # $filename is the Perl source file name from which this function was called
      if ($filename eq $0) {
         # in Perl 4, $0 is the outermost file name, so if matches $filename then the caller's source file was not DOne/REQUIREd
         $was_invoked_via_do_or_require = 0;
         # print "dbg: caller(1) filename undefined, and '$filename' eq '$0', so must be from COMMAND LINE\n";
      } else {
         $was_invoked_via_do_or_require = 1;
         # print "dbg: caller(1) filename undefined, but '$filename' ne '$0', so must be from Perl4 DO/REQUIRE\n";
      }
   }
   return $was_invoked_via_do_or_require;
}


# Here's a working example of how you might use this:
#
# Call &invoked_via_do_or_require to see if this file was loaded from the
# command line; if so, then display a help message:
if (! &invoked_via_do_or_require) {
   print "detect_do_or_require.pl  is not intended to be invoked from the command-line\n" .
         "\n" .
         "Instead, use it in another program, like this:\n" .
         "\n" .
         "  \@INC = ('.','..');'\n" .
         "  require 'detect_do_or_require.pl';\n" .
         "     ...\n" .
         "  if (\&invoked_via_do_or_require) {\n" .
         "     print \"\$0 was loaded via 'do', 'require', or 'use'\\n\";\n" .
         "  } else {\n" .
         "     print \"\$0 was invoked from command-line\\n\";\n" .
         "  }\n";
}


# return true value to 'require'
1;

__END__
