#!/usr/bin/perl

# Subroutines to create a .bmp (bitmap graphics) file, and to generate
# plots/graphs of points and functions.
#
# (Compatible with both Perl4 and Perl5.)
#
# Copyright 2010, by David A. Burton, 2010
# Cary, NC  USA
# +1-919-481-0149
# Email: http://www.burtonsys.com/email/
#
# You can use a "require" to pull this into another Perl program, like this:
#
#   @INC = ('.','..');
#   require "traceback.pl";

$| = 1;   # predefined variable. If <> 0 then each print to the console
          # will immediatly be displayed, instead of buffered.

unshift( @INC, '.' );  # make sure we fetch these modules from the current folder
require "traceback.pl";
require "detect_do_or_require.pl";
require "5x8_bitmap_charset.pl";
shift( @INC );  # restore

if (!defined $debugmode) {
   $debugmode = 0;
}


# absolute value
sub abs {
   local( $x ) = @_;
   if ($x < 0) {
      $x = - $x;
   }
   return $x;
}


# input is an integer; result is 1 (true) iff odd, null (false) if even
sub odd {
   local($i) = @_;
   return (($i & 1) == 1);
}


# Find the minimum of a list of values.  Any undefined values are ignored.
sub min {
   local( $result );
   foreach $v (@_) {
      if ((defined $v) && ((!defined $result) || ($v < $result))) {
         $result = $v;
      }
   }
   return $result;
}


# Find the maximum of a list of values.  Any undefined values are ignored.
sub max {
   local( $result );
   foreach $v (@_) {
      if ((defined $v) && ((!defined $result) || ($v > $result))) {
         $result = $v;
      }
   }
   return $result;
}


# Problem:  The built-in 'int' truncates toward zero, so int(x+0.5) only
# rounds correctly for positive arguments.
# This one rounds correctly for both positive and negative arguments.
sub round {
   local( $x ) = @_;
   if ($x >= 0) {
      $x = int( $x + 0.5 );
   } else {
      $x = int( $x - 0.5 );
   }
   return $x;
}


# The built-in 'int' truncates toward zero; this variant truncates down
# (i.e., more negative, for negative arguments).
sub floor {
   local( $x ) = @_;
   if ($x >= 0) {
      $x = int( $x );
   } else {
      local($bias) = 2 + int(-$x);
      $x = int( $bias + $x ) - $bias;
   }
   return $x;
}


# Similar to sub floor, except that the factor of which we desire an exact
# multiple is not necessarily 1.
sub floor_to_multiple_of {
   local( $x, $factor ) = @_;
   return $factor * &floor($x/$factor);
}


# calculate base 10 logarithm
sub log10 {
   local( $x ) = @_;
   if ($x <= 0) {
      &traceback;
      die "ERR: log10($x)\n";
   }
   return log($x) / log(10);
}


# Generate a 3-byte color string.  E.g., &color3b(9,8,7) returns the color
# string for Red=9, Green=8, Blue=7, which is "\007\008\009".
sub color3b {
   local( $r, $g, $b ) = @_;
   return pack("C3", $b, $g, $r);
}


# Like sub color3b, but takes one long-int parameter.  E.g., for  R/G/B = 9/8/7
# &color24( 0x090807 ) == &color3b(9,8,7) = "\007\008\009"
sub color24 {
   local( $tmp ) = @_;
   return pack("C3", ($tmp & 255), (($tmp >> 8) & 255), (($tmp >> 16) & 255));
}


# Note: BMP files contain resolution information in pixels/meter.  It is often
# ignored, but I chose to match my 1680 x 1050 ("22 inch") ViewSonic LCD monitor:
#    18.667 inches = 1680 pixels
#    1 inch = 0.254 meters;  1 Meter = ~39.3700787401575 inches
#    (1680 / 0.0254) / 18.667 = 3543 pixels / meter (= 90 dpi)
# For 72 dpi, 72 / .0254 = 2835.
# For a 600 dpi printer, you could use 600 / 0.254 = 23622.


# This is the format of the header section for a "Windows v.3" .BMP bitmap
# file (which is the most common version of a .BMP file, though several others
# are possible):
#
#                format     offset  description
#                ------     ------  --------------------------
$BMP_header_fmt = "a2" .  #  0=00H  magic number ("BM")
                  "L" .   #  2=02H  size of the BMP file in bytes, little endian
                  "L" .   #  6=06H  unused (0)
                  "L" .   # 10=0AH  offset where pixels start (54)
                  "L" .   # 14=0EH  number of additional bytes in header, starting at offset 0x0E (40)
                  "L" .   # 18=12H  width of the bitmap in pixels
                  "L" .   # 22=16H  height of the bitmap in pixels
                  "S" .   # 26=1AH  number of color panes (1)
                  "S" .   # 28=1CH  number of bits per pixel (24)
                  "L" .   # 30=1EH  BI_RGB, no compression (0)
                  "L" .   # 34=22H  size of picture in bytes (not counting this header)
                  "L" .   # 38=26H  horizontal resolution of the image (3543 pixels/meter)
                  "L" .   # 42=2AH  vertical resolution of the image (3543 pixels/meter)
                  "L" .   # 46=2EH  number of colors in palette (0)
                  "L";    # 50=32H  number of important colors (0 means all are important)
                          # 54=36H  TOTAL SIZE OF HEADER


# BMPs are stored by rows, left-to-right, usually starting with the bottom row.
# For pictures whose width is a multiple of 4 pixels (12 bytes), there's no
# padding of the rows, so the total file size = 54 + (width * height * 3).


# The bmp is represented internally by these 4 globals:
$width = $height = 0;
$header = "\000" x 54;
@pixelrows = ();


# Input is width and height of desired bitmap image, and the R/G/B color of the background.
# The color should be a 3-byte string created by &color3b or &color24.
# We initialize the header and all the pixels accordingly.
sub initialize_bmp {
   ( $width, $height, $color ) = @_;
   local( $size_of_picture_sans_header, $padlength, $i, $row, $bytwidth );
   $bytwidth = $width * 3;
   $padlength = 0;
   if (3 & $bytwidth) {
      # round up to multiple of 4 bytes
      $padlength = (4 - (3 & $bytwidth));
      $bytwidth += $padlength;
   }

   $row = ($color x $width) . ("\000" x $padlength);
   if (length($row) != $bytwidth) {
      die "ERR: miscalculated row size\n";
   }
   $size_of_picture_sans_header = $bytwidth * $height;
   $header = pack( $BMP_header_fmt, "BM",
                   54+$size_of_picture_sans_header,
                   0, 54, 40, $width, $height, 1, 24, 0,
                   $size_of_picture_sans_header,
                   3543, 3543, 0, 0 );
   @pixelrows = ();
   # print "dbg3: row = " . length($row) . " bytes = $bytwidth bytes (for width $width pixels)\n";
   for ($i=$height-1; $i >= 0; $i--) {
      $pixelrows[$i] = $row;
   }
}#initialize_bmp


# Input is the file name to create; the current BMP is written to the file.
sub write_bmp {
   local( $fname ) = @_;
   local( $i, $row );
   open( BMP, ">$fname" ) || die "ERROR: could not create \"$fname\", $!\n";
   binmode BMP;
   print BMP $header;
   # print "dbg2: width = $width, height = $height\n";
   for ($i=0; $i < $height; $i++) {
      print BMP $pixelrows[$i];
   }
   close BMP;
}


# Change one pixel in the BMP.  Inputs are X, Y, and color.
sub setpixel {
   local( $x, $y, $color ) = @_;
   if (($x < 0)  || ($y < 0) || ($x >= $width) || ($y >= $height)) {
      # print "warning, setpixel($x,$y,) but width=$width and height=$height\n";
      return;
   }
   $x *= 3;
   if (3 != length($color)) {
      traceback( 'ERR: bad color, should be 3 byte string' );
      exit(1);
   }
   $pixelrows[$y] = substr( $pixelrows[$y], 0, $x ) . $color . substr( $pixelrows[$y], $x+3 );
}


# Calculate a mixture of two colors.  E.g., &mixcolors(red,white,0.5) = pink.
sub mixcolors {
   local( $color1, $color2, $ratio ) = @_;
   local( $i, $c );
   local(@c1) = split(//,$color1);
   local(@c2) = split(//,$color2);
   local(@c3) = (0,0,0);
   for ($i=0; $i<3; $i++) {
      $c = (ord($c1[$i]) * $ratio) + (ord($c2[$i]) * (1.0 - $ratio));
      if ($c >= 255.0) {
         $c = 255;
      } else {
         $c = int( 0.5 + $c );
      }
      $c3[$i] = $c;
   }
   # print "dbg: ratio=$ratio, r1=" . ord($c1[2]) . ", r2=" . ord($c2[2]) . ", mix=$c\n";
   $c = &color3b(reverse(@c3));
   # print "dbg: mix(" . &repr_color($color1) . ', ' . &repr_color($color2) . ", $ratio) = " . &repr_color($c) . "\n";
   return $c;
}


# sub repr_color {
#    local($color) = @_;
#    local(@c) = split(//,$color);
#    local($i,$x);
#    for ($i=0; $i<3; $i++) {
#       $c[$i] = ord($c[$i]);
#    }
#    $x = '(r='.$c[2].',g='.$c[1].',b='.$c[0].')';
#    return $x;
# }
#
# $red_c = &color24(0xFF0000);
# $green_c = &color24(0x00B000);
# $lightgreen_c = &color24(0x00FF00);
# $lightgrey_c = &color24(0xD0D0D0);
# $grey_c = &color24(0xB0B0B0);  # this is the color of the graph's axes and tick-marks
# $darkgrey_c = &color24(0x505050);
# $carolina_blue_c = &color24(0x93C9FF);
# $blue_c = &color24(0x0080FF);
#
# print "blue = " . &repr_color($blue_c) . "\n";
#
#
# $tmp = mixcolors( $red_c, $blue_c, 0.9 );
# print "dbg: 90% red + 10% blue = " . &repr_color($tmp) . "\n";
#
# $tmp = mixcolors( $red_c, $blue_c, 0.5 );
# print "dbg: 50% red + 50% blue = " . &repr_color($tmp) . "\n";


# Similar to setpixel, but implements transparency.  4th input is degree of
# opacity ("alpha"), 0..1.
sub setpixel_alpha {
   local( $x, $y, $color, $alpha ) = @_;
   local( $oldcolor, $newcolor );
   if ($alpha > 0) {
       if ($alpha > 0.99) {
          &setpixel( $x, $y, $color );
       } else {
          $oldcolor = substr( $pixelrows[$y], $x*3, 3 );
          $newcolor = &mixcolors( $color, $oldcolor, $alpha );
          &setpixel( $x, $y, $newcolor );
       }
   }
}



# If coordinate is within $fuzzthresh of integer, just use the integer.
# For fully-smoothed, set $fuzzthresh to 0.
# For unsmoothed, set $fuzzthresh to >= 0.5.
$fuzzthresh = 0.01;


# Similar to setpixel, but you can set pixels at fractional coordinates, and
# it will mix colors appropriately to make it appear to be between integer
# pixel coordinates.
sub set_fractional_pixel {
   local( $x, $y, $color ) = @_;
   local( $x1, $x2, $xfrac, $y1, $y2, $yfrac );


   $x1 = &floor($x);
   $xfrac = $x - $x1;
   if ($xfrac <= $fuzzthresh) {
      $x = $x1;
      $xfrac = 0;
   } elsif ($xfrac >= (1-$fuzzthresh)) {
      $x1 = $x = $x1 + 1;
      $xfrac = 0;
   }

   $y1 = &floor($y);
   $yfrac = $y - $y1;
   if ($yfrac <= $fuzzthresh) {
      $y = $y1;
      $yfrac = 0;
   } elsif ($yfrac >= (1-$fuzzthresh)) {
      $y1 = $y = $y1 + 1;
      $yfrac = 0;
   }

   if ((0 == $xfrac) && (0 == $yfrac)){
      &setpixel($x, $y, $color);
   } else {
      &setpixel_alpha( $x1, $y1, $color, ((1-$xfrac) * (1-$yfrac)) );
      &setpixel_alpha( $x1+1, $y1, $color, ($xfrac * (1-$yfrac)) );
      &setpixel_alpha( $x1, $y1+1, $color, ((1-$xfrac) * $yfrac) );
      &setpixel_alpha( $x1+1, $y1+1, $color, ($xfrac * $yfrac) );
   }
}


# Draw a line.  Inputs are X1,Y1, X2,Y2, and color.
# Cordinates are in pixels.  Fractional pixels are okay.
sub drawline1 {
   local( $x1, $y1, $x2, $y2, $color ) = @_;
   local( $xdif, $ydif, $x, $y, $numdivs, $numpts, $i );
   $xdif = $x2 - $x1;
   $ydif = $y2 - $y1;
   $numdivs = int( 0.01 + sqrt( ($xdif * $xdif) + ($ydif * $ydif) ) );
   if ($numdivs) {
      $numpts = $numdivs + 1;
      for ($i=0; $i < $numpts; $i++) {
         $x = int( 0.5 + ($x1 + ($i / $numdivs) * ($x2 - $x1)) );
         $y = int( 0.5 + ($y1 + ($i / $numdivs) * ($y2 - $y1)) );
         &setpixel( $x, $y, $color );
      }
   }
}


# Draw a line.  Like &drawline1 except smoothed.
sub drawline3 {
   local( $x1, $y1, $x2, $y2, $color ) = @_;
   local( $xdif, $ydif, $x, $y, $numdivs, $numpts, $i );
   $xdif = $x2 - $x1;
   $ydif = $y2 - $y1;
   $numdivs = int( 0.01 + sqrt( ($xdif * $xdif) + ($ydif * $ydif) ) );
   $numpts = $numdivs + 1;
   for ($i=0; $i < $numpts; $i++) {
      $x = $x1 + (($i / $numdivs) * ($x2 - $x1));
      $y = $y1 + (($i / $numdivs) * ($y2 - $y1));
      &set_fractional_pixel( $x, $y, $color );
   }
}


# similar to &setpixel, but makes a bigger (5 pixel) diamond-shaped dot
#
#    *
#   *X*
#    *
#
sub setdot5 {
   local( $x, $y, $color ) = @_;
   &setpixel( $x, $y+1, $color );
   &setpixel( $x-1, $y, $color );
   &setpixel( $x, $y, $color );
   &setpixel( $x+1, $y, $color );
   &setpixel( $x, $y-1, $color );
}

# similar to &setpixel, but makes 3 dots horizontally
#
#   *X*
#
sub setdot3 {
   local( $x, $y, $color ) = @_;
   &setpixel( $x-1, $y, $color );
   &setpixel( $x, $y, $color );
   &setpixel( $x+1, $y, $color );
}

# similar to &setpixel, but makes a bigger 3x3 (9 pixel) square dot
#
#   ***
#   *X*
#   ***
#
sub setdot9 {
   local( $x, $y, $color ) = @_;
   local( $i, $j );
   for ($i=-1; $i<=1; $i++) {
      for ($j=-1; $j<=1; $j++) {
         &setpixel( $x+$i, $y+$j, $color );
      }
   }
}

# similar to &setpixel, but makes a bigger 5x5 (13 pixel) diamond
#    *
#   ***
#  **X**
#   ***
#    *
sub setdot13 {
   local( $x, $y, $color ) = @_;
   local( $i, $j, $absi, $absj );
   for ($i=-2; $i<=2; $i++) {
      $absi = &abs($i);
      for ($j=-2; $j<=2; $j++) {
         $absj = &abs($j);
         if (($absi+$absj) < 3) {
            &setpixel( $x+$i, $y+$j, $color );
         }
      }
   }
}

# similar to &setpixel, but makes a big 5x5 (25 pixel) square dot
#    *****
#    *****
#    **X**
#    *****
#    *****
sub setdot25 {
   local( $x, $y, $color ) = @_;
   local( $i, $j );
   for ($i=-2; $i<=2; $i++) {
      for ($j=-2; $j<=2; $j++) {
         &setpixel( $x+$i, $y+$j, $color );
      }
   }
}

# similar to &setpixel, but makes a big (21 pixel) roundish dot:
#     ***
#    *****
#    **X**
#    *****
#     ***
sub setdot21 {
   local( $x, $y, $color ) = @_;
   local( $i, $j );
   for ($i=-2; $i<=2; $i++) {
      for ($j=-2; $j<=2; $j++) {
         if (&abs($i*$j) != 4) {
            &setpixel( $x+$i, $y+$j, $color );
         }
      }
   }
}


# similar to &setpixel, but makes a big (25 pixel) diamond
# (but sub setdot pretends it is only 24 pixels):
#      *
#     ***
#    *****
#   ***X***
#    *****
#     ***
#      *
sub setdot24 {
   local( $x, $y, $color ) = @_;
   local( $i, $j, $absi, $absj );
   for ($i=-3; $i<=3; $i++) {
      $absi = &abs($i);
      for ($j=-3; $j<=3; $j++) {
         $absj = &abs($j);
         if (($absi+$absj) < 4) {
            &setpixel( $x+$i, $y+$j, $color );
         }
      }
   }
}

# similar to &setpixel, but makes a big (33 pixel) plus-shape:
#     ***
#     ***
#   *******
#   ***X***
#   *******
#     ***
#     ***
sub setdot33 {
   local( $x, $y, $color ) = @_;
   local( $i, $j );
   for ($i=-3; $i<=3; $i++) {
      for ($j=-3; $j<=3; $j++) {
         if (&abs($i*$j) < 4) {
            &setpixel( $x+$i, $y+$j, $color );
         }
      }
   }
}

# similar to &setpixel, but makes a 16-pixel triangle:
#      *
#     ***
#    **X**
#   *******
#
sub setdot16 {
   local( $x, $y, $color ) = @_;
   local( $i, $j );
   for ($i=-1; $i<=2; $i++) {
      $w = 2-$i; # 3..0
      for ($j=-$w; $j<=$w; $j++) {
         if (&abs($i*$j) < 4) {
            &setpixel( $x+$j, $y+$i, $color );
         }
      }
   }
}

# similar to &setpixel, but makes a 16-pixel upside-down triangle
# (but sub setdot pretends it is 17 pixels):
#
#   *******
#    **X**
#     ***
#      *
sub setdot17 {
   local( $x, $y, $color ) = @_;
   local( $i, $j );
   for ($i=-2; $i<=1; $i++) {
      $w = 2+$i; # 0..3
      for ($j=-$w; $j<=$w; $j++) {
         if (&abs($i*$j) < 4) {
            &setpixel( $x+$j, $y+$i, $color );
         }
      }
   }
}


# Inputs are: X, Y, color, dotsize, where dotsize can be: 1, 3, 5, 9, 13, 16, 17, 21, 24, 25, or 33.
# (Other dot sizes are rounded to a nearby allowed dot size.)
sub setdot {
   local( $x, $y, $color, $dotsize ) = @_;
   if ($dotsize > 29) {
      &setdot33( $x, $y, $color );
   } elsif ($dotsize >= 25) {
      &setdot25( $x, $y, $color );
   } elsif (24 == $dotsize) {
      &setdot24( $x, $y, $color );
   } elsif ($dotsize >= 18) {
      &setdot21( $x, $y, $color );
   } elsif ($dotsize == 17) {
      &setdot17( $x, $y, $color );
   } elsif ($dotsize == 16) {
      &setdot16( $x, $y, $color );
   } elsif ($dotsize > 10) {
      &setdot13( $x, $y, $color );
   } elsif ($dotsize > 7) {
      &setdot9( $x, $y, $color );
   } elsif ($dotsize > 3) {
      &setdot5( $x, $y, $color );
   } elsif ($dotsize == 3) {
      &setdot3( $x, $y, $color );
   } else {
      &setpixel( $x, $y, $color );
   }
}



# initialize_bmp( 200, 100, &color3b(200,0,0) );  # red 200x100
# write_bmp( "tst_red_200x100.bmp" );


# shut up '-w' compiler warning
$black_c =
 $red_c =
 $green_c =
 $lightgrey_c =
 $grey_c =
 $carolina_blue_c =
 $blue_c =
 $dark_blue_c =
 $very_light_yellow_c =
 $yellow_c =
 $orange_c =
 $brown_c =
 $purple_c =
 $pink_c = "\000\000\000";

$black_c = &color24(0);
$red_c = &color24(0xFF0000);
$green_c = &color24(0x00B000);
$lightgreen_c = &color24(0x00FF00);
$lightgrey_c = &color24(0xD0D0D0);
$grey_c = &color24(0xB0B0B0);  # this is the color of the graph's axes and tick-marks
$darkgrey_c = &color24(0x505050);
$carolina_blue_c = &color24(0x93C9FF);
$blue_c = &color24(0x0080FF);
$dark_blue_c = &color24(0x0000E0);
$very_light_yellow_c = &color24(0xFFFBD9);
$yellow_c = &color24(0xFFFF00);
$orange_c = &color24(0xFF8000);
$brown_c = &color24(0xB05800);
$purple_c = &color24(0x8000FF);
$pink_c = &color24(0xFF80C0);


# globals used for plotting:
undef $minX; undef $minY; undef $maxX; undef $maxY;
$minXaxis = $minYaxis = $maxXaxis = $maxYaxis = 0;
$XaxisPos = 0;  # $XaxisPos is the vertical bit-position of the X axis
$YaxisPos = 0;  # $YaxisPos is the horizontal bit-position of the Y axis


# For each axis, to determine its placement we need several items:
# the minimum & maximum data values, and the minimum & maximum plottable points.
# The later are what determine the placement of 0-axes, and they might represent
# slightly wider ranges of values than the former.
sub initialize_axes {
   # These represent the range of X and Y values to be plotted:
   undef $minX; undef $minY; undef $maxX; undef $maxY;
   # These determine the placement of the axes:
   $minXaxis = $minYaxis = $maxXaxis = $maxYaxis = 0;
}
&initialize_axes;


# Input is an array of 1 or more X values.
# Result is side-effect assignments setting the $minX & $maxX values, above.
# You can call this multiple times.
sub find_min_and_max_X {
   $minX = &min( $minX, @_ );
   $maxX = &max( $maxX, @_ );
}#find_min_and_max_X


# Input is an array of 1 or more Y values.
# Result is side-effect assignments setting the $minY & $maxY values, above.
# You can call this multiple times.
sub find_min_and_max_Y {
   $minY = &min( $minY, @_ );
   $maxY = &max( $maxY, @_ );
}#find_min_and_max_Y


# Input is a lst of X,Y pairs: x1,y1, x2,y2, x3,y3, etc.
# Set the minX, maxX, minY, and maxY globals accordingly.
sub find_min_and_max_XY {
   local( @x, @y );
   if (!&odd($#_)) { die "ERR: called find_min_and_max_XY() w/ odd number of parameters\n"; }
   while ($#_ > 0) {
      push( @x, shift( @_ ) );
      push( @y, shift( @_ ) );
   }
   &find_min_and_max_X( @x );
   &find_min_and_max_Y( @y );
}


# Given the min/max range for X, calculate the adjusted min/max values for
# inclusion of Y-axis, and also the scaling and offset values.
sub calc_adjusted_X_range {
   $minXaxis = $minX;
   $maxXaxis = $maxX;
   if (($minXaxis > 0) && ($maxXaxis > (4 * $minX))) {
      # if range of X-values is close to zero, include zero so we get an axis
      $minXaxis = 0;
   } elsif (($maxXaxis < 0) && ($minXaxis < (4 * $maxX))) {
      $maxXaxis = 0;
   }
   $rangeX = $maxXaxis - $minXaxis;
   if (!$rangeX) {
      $rangeX = 1;
   }
   if ((!defined $width) || ($width <= 0)) {
      &traceback;
      die "ERR: sub calc_adjusted_X_range was called before sub initialize_bmp\n";
   }
   $scaleX = 0.98 * ($width / $rangeX);
   $ofsX = $minXaxis - (0.01 * $rangeX);
   if ($debugmode) {
      print "dbg: ofsX = $ofsX, minXaxis = $minXaxis; maxXaxis = $maxXaxis -> ";
   }
   if ($ofsX < $minXaxis) {
      $minXaxis = $ofsX;
   }
   if ((($width/$scaleX) + $ofsX) > $maxXaxis) {
      $maxXaxis = ($width/$scaleX) + $ofsX;
   }
   if ($debugmode) {
      print "minXaxis=$minXaxis, maxXaxis=$maxXaxis\n";
   }
}

# Given the min/max range for Y, calculate the adjusted min/max values for
# inclusion of C-axis, and also the scaling and offset values.
sub calc_adjusted_Y_range {
   $minYaxis = $minY;
   $maxYaxis = $maxY;
   if (($minYaxis > 0) && ($maxYaxis > (4 * $minY))) {
      # if range of Y-values is close to zero, include zero so we get an axis
      $minYaxis = 0;
   } elsif (($maxYaxis < 0) && ($minYaxis < (4 * $maxY))) {
      $maxYaxis = 0;
   }
   $rangeY = $maxYaxis - $minYaxis;
   if (!$rangeY) {
      $rangeY = 1;
   }
   $scaleY = 0.98 * ($height / $rangeY);
   $ofsY = $minYaxis - (0.01 * $rangeY);
   # print "dbg: ofsY = $ofsY, minYaxis = $minYaxis; maxYaxis = $maxYaxis -> ";
   if ($ofsY < $minYaxis) {
      $minYaxis = $ofsY;
   }
   if ((($height/$scaleY) + $ofsY) > $maxYaxis) {
      $maxYaxis = ($height/$scaleY) + $ofsY;
   }
   # print "$maxYaxis\n";
}

# Given the min/max ranges for the X and Y values, calculate the adjusted
# min/max values for inclusion of axes, and also the scaling and offset values.
sub calc_adjusted_XY_ranges {
   &calc_adjusted_X_range;
   &calc_adjusted_Y_range;
}


# Convert from X coordinate to horizontal pixel position
sub x2pixel {
   if (!defined $ofsX) {
      traceback( 'ERR: ofsX is undefined' );
      exit(1);
   }
   if (!defined $scaleX) {
      traceback( 'ERR: scaleX is undefined' );
      exit(1);
   }
   if (!defined $_[0]) {
      traceback( 'ERR: undefined input parameter' );
      exit(1);
   }
   return int((($_[0] - $ofsX) * $scaleX) + 0.5);
}


# Convert from Y coordinate to vertical pixel position
sub y2pixel {
   if (!defined $ofsY) {
      traceback( 'ERR: ofsY is undefined' );
      exit(1);
   }
   if (!defined $scaleY) {
      traceback( 'ERR: scaleY is undefined' );
      exit(1);
   }
   if (!defined $_[0]) {
      traceback( 'ERR: undefined input parameter' );
      exit(1);
   }
   return int((($_[0] - $ofsY) * $scaleY) + 0.5);
}


# reverse of x2pixel  (currently unused)
sub pixel2x {
   return ($_[0] / $scaleX) + $ofsX;
}


# reverse of y2pixel  (currently unused)
sub pixel2y {
   return ($_[0] / $scaleY) + $ofsY;
}


# Draw a line.  Like &drawline1 except that cordinates are in the units
# being graphed, instead of in pixels.
sub drawline2 {
   local( $x1, $y1, $x2, $y2, $color ) = @_;
   $x1 = &x2pixel($x1);
   $y1 = &y2pixel($y1);
   $x2 = &x2pixel($x2);
   $y2 = &y2pixel($y2);
   &drawline1( $x1, $y1, $x2, $y2, $color );
}


# Draw the 0-axes
sub plot_axes {
   # Plot the X axis:
   if (($minYaxis <= 0) && ($maxYaxis >= 0)) {
      $XaxisPos = &y2pixel(0);
   } else {
      $XaxisPos = 0;
   }
   for ($i = 0; $i < $width; $i++) {
      &setpixel( $i, $XaxisPos, $grey_c );
   }
   # Plot the Y axis:
   if (($minXaxis <= 0) && ($maxXaxis >= 0)) {
      $YaxisPos = &x2pixel(0);
   } else {
      $YaxisPos = 0;
   }
   for ($i = 0; $i < $height; $i++) {
      &setpixel( $YaxisPos, $i, $grey_c );
   }
}


# draw a single character at specified X & Y dot-position
sub drawch {
   local( $x, $y, $char, $color ) = @_;
   local( $ch_bits ) = &get_bitmap_for_char( $char );  # 5x8 character bitmap
   local( $i, $j, $row );
   for ($i=0; $i<8; $i++) {
      $row = ord(substr($ch_bits,$i,1));
      for ($j=0; $j<=5; $j++) {
         if (($row & 0x10) != 0) {
            &setpixel( $x+$j-2, $y-$i+4, $color );
         }
         $row <<= 1;
      }
   }
}


# how many pixels wide will the string be, when displayed?
sub pixellen_str {
   local( $str ) = @_;
   if ('' eq $str) {
      return 0;
   } else {
      return length($str) * 7 - 2;
   }
}


# Draw a string starting at specified X & Y dot-position.
# The first character will be centered at (X,Y) and subsequent characters will be to the right.
sub drawstr {
   local( $x, $y, $str, $color ) = @_;
   local( $i, $ch );
   for ($i=0; $i<length($str); $i++) {
      $ch = substr($str,$i,1);
      &drawch( $x, $y, $ch, $color );
      $x += 7;
   }
}


# Like drawstr, except that if the string doesn't entirely fit on the graph it
# won't be drawn at all
sub draw_entire_str {
   local( $x, $y, $str, $color ) = @_;
   local($l) = &pixellen_str($str);
   if ((($y+4) < $height) && (($y-3) >= 0) && (($x-2) >= 0) && (($x+$l-3) < $width)) {
      &drawstr( $x, $y, $str, $color );
   }
}


# Inputs are the min & max (representing the range of displayable values),
# and the desired approximate number of ticks.
# Ticks must be at multiples of 5, 10, 20, 25, 50, 100, 200, 250, 500, 1000,
# 2000, etc., so the number of ticks is unlikely to be exactly the requested
# number of ticks, but it will be fairly close.
# Outputs are the first tick value and the increment between ticks.
sub calc_tick_increment {
   local( $minx, $maxx, $requested_tickcount ) = @_;
   local( $range, $range_per_tick, $ticksize, $firsttick,
          $normalizer, $normalized_range );
   $range = $maxx - $minx;
   $range_per_tick = $range / $requested_tickcount;
   $normalizer = 10 ** &floor(&log10($range_per_tick));  # a power of 10  (0.1, 1, 10, 100, etc.)
   $normalized_range = $range_per_tick / $normalizer;  # should be between 1 and 9.9999
   if ($normalized_range <= 1.5) {
      $ticksize = 1;
   } elsif ($normalized_range <= 2.25) {
      $ticksize = 2;
   } elsif ($normalized_range <= 3.75) {
      $ticksize = 2.5;
   } else {
      $ticksize = 5;
   }
   $ticksize *= $normalizer;
   # print "dbg: range/tick=$range_per_tick, normalizer=$normalizer, normalized_range=$normalized_range, ticksize=$ticksize\n";
   $firsttick = &floor_to_multiple_of( $minx, $ticksize );
   # print "dbg: minx=$minx, minx/normalizer=" . ($minx/$normalizer) . " floor -> " . &floor($minx/$normalizer) . ", x normalizer -> firsttick = $firsttick\n";
   if ($firsttick < $minx) {
      $firsttick += $ticksize;
      # print "dbg: firsttick bumped up to $firsttick\n";
   }
   return ($firsttick, $ticksize);
}


# Draw the axis ticks or grid-lines, and label them.
# Pass $grid==1 to draw full grid lines instead of just axis ticks.
sub add_ticks {
   local( $grid ) = @_;
   local( $lo, $hi, $incr, $bitx, $bity, $labl, $labl_len, $up_by, $over_by );
   ($firstXtick, $Xticksize) = &calc_tick_increment( $minXaxis, $maxXaxis, 10 );
   ($firstYtick, $Yticksize) = &calc_tick_increment( $minYaxis, $maxYaxis, 10 );

   # Plot the X axis ticks (vertical grid lines):
   if ($grid) {
      # where we place the labels varies slightly depending on whether we're
      # drawing ticks or grid lines.  Also, ticks are drawn with 7 consecutive
      # pixels, but grid lines are drawn with just every 3rd pixel.
      $lo = 0;
      $hi = $height-1;
      $incr = 3;
   } else {
      $lo = $XaxisPos - 3;
      $hi = $XaxisPos + 3;
      $incr = 1;
   }
   for ($x=$firstXtick; $x <= $maxXaxis; $x += $Xticksize) {
      $bitx = &x2pixel($x);
      for ($i = $lo; $i <= $hi; $i += $incr) {
         &setpixel( $bitx, $i, $grey_c );
      }
      # now add the labels:
      if (&abs($x) > ($Xticksize/8)) {  # we don't label the 0-point
         $labl = ''.$x;
         $labl_len = 0;
         if (length($labl) > 1) {
            $labl_len = ((length($labl)-1) * 7);
            # width of 2nd-Nth string digits in display-bits
         }
         $over_by =  - int( $labl_len / 2 );
         if ($grid) {
            $up_by = 6;
         } else {
            $up_by = 9;
         }
         &draw_entire_str( $bitx+$over_by, $XaxisPos+$up_by, $labl, $black_c );
      }
   }

   # Plot the Y axis ticks (horizontal grid lines)
   if ($grid) {
      $lo = 0;
      $hi = $width-1;
   } else {
      $lo = $YaxisPos - 3;
      $hi = $YaxisPos + 3;
   }
   for ($y=$firstYtick; $y <= $maxYaxis; $y += $Yticksize) {
      $bity = &y2pixel($y);
      for ($i = $lo; $i <= $hi; $i += $incr) {
         &setpixel( $i, $bity, $grey_c );
      }
      # now add the labels:
      if (&abs($y) > ($Yticksize/8)) {  # we don't label the 0-point
         if ($grid) {
            $up_by = 5; # so it doesn't overwrite the grid line
            $over_by = 6;
         } else {
            $up_by = 0;
            $over_by = 9;
         }
         &draw_entire_str( $YaxisPos+$over_by, $bity+$up_by, ''.$y, $black_c );
      }
   }
}


# input is a color, a dot size, an array of X values, and an array of Y values.
sub plot_points {
   if (!&odd($#_)) { die "ERR: called plot_points() w/ odd number of parameters\n"; }
   local( $color ) = shift @_;
   local( $numpts ) = ($#_ / 2);
   local( $dotsize ) = shift @_;
   local( @x ) = splice( @_, 0, $numpts );
   local( @y ) = @_;
   local( $i );
   # Plot the data points:
   for ($i=0; $i<$numpts; $i++) {
      &setdot( &x2pixel($x[$i]), &y2pixel($y[$i]), $color, $dotsize );
   }
}#plot_points


# input is a color, a dot size, and a list of X,Y pairs.
sub plot_points2 {
   if (!&odd($#_)) { die "ERR: called plot_points2() w/ odd number of parameters\n"; }
   local( $color ) = shift @_;
   local( $dotsize ) = shift @_;
   local( $x, $y );
   # Plot the data points:
   while ($#_ > 0) {
      $x = shift @_;
      $y = shift @_;
      &setdot( &x2pixel($x), &y2pixel($y), $color, $dotsize );
   }
}#plot_points2


# Input is a minimum X value, followed by a maximum X value,
# followed by an eval-able expression on $x (as a string).
# Result is a list of X,Y pairs, suitable for plotting.
sub gen_function_plot {
   local( $minX, $maxX, $function_of_x ) = @_;
   local( $i, $x, $rangeX, $y, $numpts, @xy );
   $rangeX = $maxX - $minX;
   # print "dbg2: minX=$minX, maxX=$maxX, range=$rangeX\n";
   @xy = ();
   # $numpts = $width * 3;   # makes a thicker line, but it looks funny
   $numpts = $width;
   for ($i = 0; $i <$numpts; $i++) {
      $x = $minX + ($i / $numpts) * $rangeX;
      $y = eval($function_of_x);
      push( @xy, $x );
      push( @xy, $y );
   }
   return @xy;
}



sub demo_plot {

   &initialize_bmp( 700, 500, $very_light_yellow_c );

   @x1 = (   -70, -65, -60, 45 );
   @y1 = (   2, 4, 4.2, 4.4  );
   @x2 = (        80, 90, 100,  110,  120,  128 );
   @y2 = (        3, 3,  3, 2.75, 2.5, 2.25 );
   &find_min_and_max_X( @x1 );
   &find_min_and_max_Y( @y1 );
   &find_min_and_max_X( @x2 );
   &find_min_and_max_Y( @y2 );
   &find_min_and_max_XY( 0, 0 );  # ensure that 0-axes get plotted
   &find_min_and_max_Y( 7 ); # make it a bit taller
   &calc_adjusted_X_range;
   # print "dbg: minX=$minX, maxX=$maxX, minY=$minY, maxY=$maxY, maxYaxis=$maxYaxis (before gen_function_plot)\n";
   @xy = &gen_function_plot( $minX, $maxX, '(1 + (0.01*$x) + (0.00015*($x**2)))' );
   &find_min_and_max_XY( @xy );
   # print "dbg: minX=$minX, maxX=$maxX, minY=$minY, maxY=$maxY, maxYaxis=$maxYaxis (after gen_function_plot)\n";
   &calc_adjusted_Y_range;
   # print "dbg: minX=$minX, maxX=$maxX, minY=$minY, maxY=$maxY, maxYaxis=$maxYaxis (after calc_adjusted_Y_range)\n";

   # This is optional; it is just to draw the tips of the curve, with X < minX or X > maxX:
   @xy = &gen_function_plot( $minXaxis, $maxXaxis, '(1 + (0.01*$x) + (0.00015*($x**2)))' );
   # print "dbg: minX=$minX, maxX=$maxX, minY=$minY, maxY=$maxY, maxYaxis=$maxYaxis (after 2nd gen_function_plot)\n";

   &plot_axes;
   &add_ticks(1);
   &plot_points( $red_c, 25, @x1, @y1 );
   &plot_points( $black_c, 24, @x2, @y2 );
   &plot_points2( $green_c, 1, @xy );

   # Try out all the different dot styles & sizes
   # dotsize can be: 1, 5, 9, 13, 16, 17, 21, 24, 25, or 33.
   &drawstr( &x2pixel(11), &y2pixel(6.6), "Examples of the ten dot-styles:", $purple_c );
   &plot_points2( $black_c, 1, 11, 6.1 );
   &plot_points2( $green_c, 5, 21, 6.1 );
   &plot_points2( $blue_c, 9, 31, 6.1 );
   &plot_points2( $red_c, 13, 41, 6.1 );
   &plot_points2( $green_c, 16, 51, 6.1 );
   &plot_points2( $blue_c, 17, 61, 6.1 );
   &plot_points2( $red_c, 21, 71, 6.1 );
   &plot_points2( $green_c, 24, 81, 6.1 );
   &plot_points2( $blue_c, 25, 91, 6.1 );
   &plot_points2( $red_c, 33, 101, 6.1 );
   local($bity) = &y2pixel(6.3);
   &drawstr( &x2pixel(11), $bity, 1, $black_c );
   &drawstr( &x2pixel(21), $bity, 5, $green_c );
   &drawstr( &x2pixel(31), $bity, 9, $blue_c );
   &drawstr( &x2pixel(41)-3, $bity, 13, $red_c );
   &drawstr( &x2pixel(51)-3, $bity, 16, $green_c );
   &drawstr( &x2pixel(61)-3, $bity, 17, $blue_c );
   &drawstr( &x2pixel(71)-3, $bity, 21, $red_c );
   &drawstr( &x2pixel(81)-3, $bity, 24, $green_c );
   &drawstr( &x2pixel(91)-3, $bity, 25, $blue_c );
   &drawstr( &x2pixel(101)-3, $bity, 33, $red_c );

   # draw some lines
   $color = $black_c;
   $y = 150;
   for ($i=0; $i<=50; $i+=5) {
      $x1 = 150 - $i;
      $y1 = $y + $i;
      $x2 = 150 + $i;
      $y2 = ($y+100) - $i;
      # print "($x1,$y1) ($x2,$y2)\n";
      &drawline1( $x1,$y1, $x2,$y2, $color );
      if ($color eq $red_c) {
         $color = $green_c;
      } elsif ($color eq $green_c) {
         $color = $blue_c;
      } else {
         $color = $red_c;
      }
   }

   # draw some smooth lines
   $color = $black_c;
   $y = 250;
   for ($i=0; $i<=50; $i+=5) {
      $x1 = 150 - $i;
      $y1 = $y + $i;
      $x2 = 150 + $i;
      $y2 = ($y+100) - $i;
      # print "($x1,$y1) ($x2,$y2)\n";
      &drawline3( $x1,$y1, $x2,$y2, $color );
      if ($color eq $red_c) {
         $color = $green_c;
      } elsif ($color eq $green_c) {
         $color = $blue_c;
      } else {
         $color = $red_c;
      }
   }

   # draw some smooth-ish lines
   $fuzzthresh = 0.2;
   $color = $black_c;
   $y = 350;
   for ($i=0; $i<=50; $i+=5) {
      $x1 = 150 - $i;
      $y1 = $y + $i;
      $x2 = 150 + $i;
      $y2 = ($y+100) - $i;
      # print "($x1,$y1) ($x2,$y2)\n";
      &drawline3( $x1,$y1, $x2,$y2, $color );
      if ($color eq $red_c) {
         $color = $green_c;
      } elsif ($color eq $green_c) {
         $color = $blue_c;
      } else {
         $color = $red_c;
      }
   }

   # &drawstr( 5, 250, '!"#$%&\'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~', $black_c );

   &write_bmp( "tst_tmpplot_700x500.bmp" );
   local($fsize) = -s "tst_tmpplot_700x500.bmp";
   print "Created demo plot: 'tst_tmpplot_700x500.bmp' ($fsize bytes)\n";
   print "About to tell OS to open tst_tmpplot_700x500.bmp .";
   sleep 0.75;
   print " .";
   sleep 0.75;
   print " .\n";
   `dir tst_tmpplot_700x500.bmp`;
   `tst_tmpplot_700x500.bmp`;
   sleep 1;
   print "Back from opening tst_tmpplot_700x500.bmp!\n";
   print "Would you like to delete 'tst_tmpplot_700x500.bmp'? (y/n) ";
   local($ch) = getc;
   if ($ch =~ /^y/i) {
      print "Deleting tst_tmpplot_700x500.bmp .";
      sleep 0.75;
      print " .";
      sleep 0.75;
      print " .\n";
      unlink( "tst_tmpplot_700x500.bmp" );
   }
   print "Done.\n";
   sleep 1;
}


# if invoked from command line, make a demo plot
if (! &invoked_via_do_or_require) {
   &demo_plot;
}


1;

__END__

