#!/usr/bin/perl

# Subroutine to solve three equations for three unknowns.
#
# (Compatible with both Perl4 and Perl5.)
#
# This file is UNCOPYRIGHTED, by David A. Burton, 2010
# Cary, NC  USA
# +1-919-481-0149
# Email: http://www.burtonsys.com/email/
#
# You should use a "require" to pull this into another Perl program, like this:
#
#   use bignum;  # (optional, but highly recommended; available in Perl 5.008 & later)
#   unshift( @INC, '.' );
#   require "threeunknowns.pl";
#   no bignum;  # (after loading it you may disable use of huge-precision/slow arithmetic, if you wish)
#

# TLIB Version Control fills in the version information for us:
$version_str = "";
#--=>keyflag<=-- "&(#)%n, version %v, %d "
$version_str = "&(#)threeunknowns.pl, version 3, 25-May-10 ";


$| = 1;   # predefined variable. If <> 0 then each print to the console
          # will immediatly be displayed, instead of buffered.

unshift( @INC, '.' );  # make sure we fetch these modules from the current folder
require "detect_do_or_require.pl";  # define &invoked_via_do_or_require
require "traceback.pl";  # define &traceback
shift( @INC );  # restore


# Based on http://www.1728.com/unknwn3.htm  (in JavaScript)  by Robert
# Gagnon, a/k/a wolf_meister.  (Actually, there's hardly anything left of
# his code, but I found it instructive, so I'm leaving this credit here.)
#
# Given the 12 coefficients (A1,B1,C1,D1,A2,...,D3) of these 3 equations:
#
#   A1*x + B1*y + C1*z = D1
#   A2*x + B2*y + C2*z = D2
#   A3*x + B3*y + C3*z = D3
#
# Solve for the 3 unknowns (x,y,z):
#
#   // calculate the "delta" determinant
#   delta = (A1*B2*C3)+(B1*C2*A3)+(C1*A2*B3)-(C1*B2*A3)-(A1*C2*B3)-(B1*A2*C3)
#
#   // calculate the X numerator & answer
#   xnum = (D1*B2*C3)+(B1*C2*D3)+(C1*D2*B3)-(C1*B2*D3)-(D1*C2*B3)-(B1*D2*C3)
#   x = xnum/delta
#
#   // calculate the Y numerator & answer
#   ynum = (A1*D2*C3)+(D1*C2*A3)+(C1*A2*D3)-(C1*D2*A3)-(A1*C2*D3)-(D1*A2*C3)
#   y = ynum/delta
#
#   // calculate the Z numerator & answer
#   znum = (A1*B2*D3)+(B1*D2*A3)+(D1*A2*B3)-(D1*B2*A3)-(A1*D2*B3)-(B1*A2*D3)
#   z = znum/delta


# Given the 12 coefficients of the 3 linear functions on ($x,$y,$z), calculate
# and return ($x,$y,$z)
sub three_unknowns {
   local( $a1,$b1,$c1,$d1, $a2,$b2,$c2,$d2, $a3,$b3,$c3,$d3 ) = @_;
   if ($#_ != 11) {
      &traceback;
      die "ERR: sub three_unknowns called with " . (1+$#_) . " parameters instead of 12\n";
   }
   local($i);
   for ($i=0; $i<$#_; $i++) {
      if (!defined $_[$i]) {
         die "ERR: argument no. " . (1+$i) . " to sub three_unknowns is undefined\n";
      }
   }
   local($delta,$xnum,$ynum,$znum,$x,$y,$z);
   # calculate the "delta" determinant
   $delta = ($a1 * $b2 * $c3) + ($b1 * $c2 * $a3) + ($c1 * $a2 * $b3) - ($c1 * $b2 * $a3) - ($a1 * $c2 * $b3) - ($b1 * $a2 * $c3);
   if (0 == $delta) {
      &traceback( "ERROR: division-by-zero!" );
      $x = $y = $z = 0;
   } else {
      # calculate the X numerator & answer
      $xnum = ($d1 * $b2 * $c3) + ($b1 * $c2 * $d3) + ($c1 * $d2 * $b3) - ($c1 * $b2 * $d3) - ($d1 * $c2 * $b3) - ($b1 * $d2 * $c3);
      $x = $xnum/$delta;
      # calculate the Y numerator & answer
      $ynum = ($a1 * $d2 * $c3) + ($d1 * $c2 * $a3) + ($c1 * $a2 * $d3) - ($c1 * $d2 * $a3) - ($a1 * $c2 * $d3) - ($d1 * $a2 * $c3);
      $y = $ynum/$delta;
      # calculate the Z numerator & answer
      $znum = ($a1 * $b2 * $d3) + ($b1 * $d2 * $a3) + ($d1 * $a2 * $b3) - ($d1 * $b2 * $a3) - ($a1 * $d2 * $b3) - ($b1 * $a2 * $d3);
      $z = $znum/$delta;
   }
   return ($x,$y,$z);
}


# if invoked from command line, print an error message
if (! &invoked_via_do_or_require) {
   print "$0  is intended to be loaded via 'require'.  See comments in\n" .
         "the source code file for instructions, and this example:\n";

   print "\n" .
   "sub three_unknowns is a subroutine to solve 3 linear equations for 3 unknowns.\n" .
   "\n" .
   "EXAMPLE: For the equations:\n" .
   "\n" .
   "   2X  + 3Y +  4Z = 119\n" .
   "   5X  - 6Y +  7Z =  80\n" .
   "   8X  + 9Y + 10Z = 353\n" .
   "\n" .
   "You would pass:\n" .
   "\n" .
   "   2,  3,  4, 119,\n" .
   "   5, -6,  7,  80,\n" .
   "   8,  9, 10, 353\n" .
   "\n" .
   "And get the result:\n" .
   "\n" .
   "   X=12 Y=13 Z=14\n" .
   "\n" .
   "Your code looks like this:\n" .
   "\n" .
   "   (\$x,\$y,\$z) = \&three_unknowns( 2, 3, 4, 119,  5, -6, 7, 80,  8, 9, 10, 353 );\n" .
   "   print \"   X=\$x Y=\$y Z=\$z\\n\";\n" .
   "\n" .
   "which prints...\n" .
   "\n";

   ($x,$y,$z) = &three_unknowns( 2, 3, 4, 119,  5, -6, 7, 80,  8, 9, 10, 353 );
   print "   X=$x Y=$y Z=$z\n";

   exit 1;
}


1;

__END__

