#!/usr/bin/perl

# Subroutine to solve two equations for two unknowns.
#
# (Compatible with both Perl4 and Perl5.)
#
# This file is UNCOPYRIGHTED, by David A. Burton, 2010
# Cary, NC  USA
# +1-919-481-0149
# Email: http://www.burtonsys.com/email/
#
# You should use a "require" to pull this into another Perl program, like this:
#
#   use bignum;  # (optional, but highly recommended; available in Perl 5.008 & later)
#   unshift( @INC, '.' );
#   require "twounknowns.pl";
#   no bignum;  # (after loading it you may disable use of huge-precision/slow arithmetic, if you wish)
#

# TLIB Version Control fills in the version information for us:
$version_str = "";
#--=>keyflag<=-- "&(#)%n, version %v, %d "
$version_str = "&(#)twounknowns.pl, version 2, 26-May-10 ";


$| = 1;   # predefined variable. If <> 0 then each print to the console
          # will immediatly be displayed, instead of buffered.

unshift( @INC, '.' );  # make sure we fetch these modules from the current folder
require "detect_do_or_require.pl";  # define &invoked_via_do_or_require
require "traceback.pl";  # define &traceback
shift( @INC );  # restore


# Given the 6 coefficients (A1,B1,C1,A2,B2,C2) of these 2 equations:
#
#   A1*x + B1*y = C1
#   A2*x + B2*y = C2
#
# Solve for the 2 unknowns (x,y):
#
#   // calculate the "delta" determinant
#   delta = (A1*B2)-(B1*A2)
#
#   // calculate the X numerator & answer
#   xnum = (B2*C1)-(B1*C2)
#   x = xnum/delta
#
#   // calculate the Y numerator & answer
#   ynum = (A1*C2)-(A2*C1)
#   y = ynum/delta
#



sub aprox_equal {
   local( $a, $b ) = @_;
   local( $result );
   if ((0==$a) || (0==$b)) {
      $result = 1;  # zero is a funny case, so don't even check
   } elsif ($b > 0) {
      $result = ($a > (0.99999 * $b)) && ($a < (1.00001 * $b));
   } else {
      $result = ($a < (0.99999 * $b)) && ($a > (1.00001 * $b));
   }
   return $result;
}


# Given the 6 coefficients of the 2 linear functions on ($x,$y), calculate
# and return ($x,$y)
sub two_unknowns {
   local( $a1,$b1,$c1, $a2,$b2,$c2 ) = @_;
   if ($#_ != 5) {
      &traceback;
      die "ERR: sub two_unknowns called with " . (1+$#_) . " parameters instead of 6\n";
   }
   local($i);
   for ($i=0; $i<$#_; $i++) {
      if (!defined $_[$i]) {
         &traceback;
         die "ERR: argument no. " . (1+$i) . " to sub two_unknowns is undefined\n";
      }
   }
   local($delta,$xnum,$ynum,$x,$y);
   # calculate the "delta" determinant
   $delta = ($a1 * $b2) - ($b1 * $a2);
   if (0 == $delta) {
      warn "ERROR: division-by-zero in sub two_unknowns!";
      &traceback;
      $x = $y = 0;
   } else {
      # calculate the X numerator & answer
      $xnum = ($b2 * $c1) - ($b1 * $c2);
      $x = $xnum/$delta;
      # calculate the Y numerator & answer
      $ynum = ($a1 * $c2) - ($a2 * $c1);
      $y = $ynum/$delta;
      # check it (this is debug code, and can be removed):
      if (   (!&aprox_equal( (($a1 * $x) + ($b1 * $y)), $c1 ))
          || (!&aprox_equal( (($a2 * $x) + ($b2 * $y)), $c2 )) ) {
         &traceback;
         die "ERR: sub two_unknowns got wrong result:\n" .
             "  x=$x, y=$y,\n" .
             "  a1=$a1, b1=$b1, c1=$c1,\n" .
             "  a2=$a2, b2=$b2, c2=$c2,\n" .
             "  ((a1*x)+(b1*y))=" . (($a1*$x)+($b1*$y)) . " != c1=$c1    or\n" .
             "  ((a2*x)+(b2*y))=" . (($a2*$x)+($b2*$y)) . " != c2=$c2\n";
      }
   }
   return ($x,$y);
}


# if invoked from command line, print an error message
if (! &invoked_via_do_or_require) {
   print "$0  is intended to be loaded via 'require'.  See comments in\n" .
         "the source code file for instructions, and this example:\n";

   print "\n" .
   "sub two_unknowns is a subroutine to solve 2 linear equations for 2 unknowns.\n" .
   "\n" .
   "\n" .
   "EXAMPLE: For the equations:\n" .
   "\n" .
   "   2X  + 3Y =  63\n" .
   "   5X  - 6Y = -18\n" .
   "\n" .
   "You would pass:\n" .
   "\n" .
   "   2,  3,  63,\n" .
   "   5, -6, -18,\n" .
   "\n" .
   "And get the result:\n" .
   "\n" .
   "   X=12 Y=13\n" .
   "\n" .
   "Your code looks like this:\n" .
   "\n" .
   "   (\$x,\$y) = \&two_unknowns( 2, 3, 63,  5, -6, -18 );\n" .
   "   print \"   X=\$x Y=\$y\\n\";\n" .
   "\n" .
   "which prints...\n" .
   "\n";

   ($x,$y) = &two_unknowns( 2, 3, 63,  5, -6, -18 );
   print "   X=$x Y=$y\n";

   print "\n" .
   "\n" .
   "EXAMPLE \#2:\n" .
   "\n" .
   "   (\$x,\$y) = \&two_unknowns( 1, 2, 3,  4, 5, 6 );\n" .
   "   print \"   X=\$x Y=\$y\\n\";  \# prints  X=-1 Y=2\n" .
   "\n" .
   "prints...\n" .
   "\n";

   ($x,$y) = &two_unknowns( 1, 2, 3,  4, 5, 6 );
   print "   X=$x Y=$y\n";  # prints  X=-1 Y=2

   print "\n" .
   "\n" .
   "EXAMPLE \#2:\n" .
   "\n" .
   "   (\$x,\$y) = \&two_unknowns( 7, 8, 10,  10, 13, 21 );\n" .
   "   print \"   X=\$x Y=\$y\\n\";  \# prints  X=-3.45454545454545 Y=4.27272727272727\n" .
   "\n" .
   "prints...\n" .
   "\n";
   ($x,$y) = &two_unknowns( 7, 8, 10,  10, 13, 21 );
   print "   X=$x Y=$y\n";  # prints  X=-3.45454545454545 Y=4.27272727272727

   print "\n" .
   "\n" .
   "EXAMPLE \#3:\n" .
   "\n" .
   "   use bignum;\n" .
   "   (\$x,\$y) = \&two_unknowns( 7, 8, 10,  10, 13, 21 );\n" .
   "   print \"   X=\$x Y=\$y\\n\";\n" .
   "   no bignum;\n" .
   "\n" .
   "prints...\n" .
   "\n" .
   "   X=-3.454545454545454545454545454545454545455 Y=4.272727272727272727272727272727272727273\n" .
   "\n";
  # use bignum;
  # ($x,$y) = &two_unknowns( 7, 8, 10,  10, 13, 21 );
  # print "   X=$x Y=$y\n";
  # no bignum;

   exit 1;
}


1;

__END__

